<?php


namespace Mnv\Http\Handler;

use Mnv\Http\Response;
use Mnv\Http\JsonResponse;
use Mnv\Core\Collections\Str;
use Mnv\Core\Collections\Traits\Macroable;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpFoundation\StreamedResponse;

/**
 * Class ResponseFactory
 * @package Mnv\Http
 */
class ResponseFactory
{

    use Macroable;

    /**
     * Return a new response from the application.
     *
     * @param string|null $content
     * @param int $status
     * @param  array  $headers (optional)
     * @return Response
     */
    public function make(string $content = null, int $status = 200, array $headers = []): Response
    {
        return new Response($content, $status, $headers);
    }

    /**
     * Return a new JSON response from the application.
     *
     * @param array $data
     * @param int $status
     * @param array $headers (optional)
     * @param array $options (optional)
     * @return JsonResponse
     */
    public function json(array $data = [], int $status = 200, array $headers = [], $options = 0): JsonResponse
    {
        return new JsonResponse($data, $status, $headers, $options);
    }

    /**
     * Create a new streamed response instance.
     *
     * @param  \Closure  $callback
     * @param  int  $status
     * @param  array  $headers
     * @return \Symfony\Component\HttpFoundation\StreamedResponse
     */
    public function stream($callback, $status = 200, array $headers = []): StreamedResponse
    {
        return new StreamedResponse($callback, $status, $headers);
    }

    /**
     * Create a new file download response.
     *
     * @param  \SplFileInfo|string  $file
     * @param  string  $name
     * @param  array  $headers
     * @param  null|string  $disposition
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function download($file, $name = null, array $headers = [], $disposition = 'attachment'): BinaryFileResponse
    {
        $response = new BinaryFileResponse($file, 200, $headers, true, $disposition);

        if (! is_null($name)) {
            return $response->setContentDisposition($disposition, $name, str_replace('%', '', Str::ascii($name)));
        }

        return $response;
    }
}