# Коллекции

- [Введение](#introduction)
    - [Создание коллекций](#creating-collections)
- [Доступные методы](#available-methods)
- [Higher Order Messages](#higher-order-messages)
- [Ленивые коллекции](#lazy-collections)
    - [Введение](#lazy-collection-introduction)
    - [Создание ленивых коллекций](#creating-lazy-collections)
    - [Перечислимый контракт](#the-enumerable-contract)
    - [Ленивые методы сбора](#lazy-collection-methods)

<a name="introduction"></a>
## Введение

Класс `Mnv\Core\Collections\Collection` предоставляет беглую удобную оболочку для работы с массивами данных. 
Например, проверьте следующий код. Мы воспользуемся `collect` помощником для создания нового экземпляра коллекции из массива, запустим функцию `strtoupper` для каждого элемента, а затем удалим все пустые элементы:

    $collection = collect(['taylor', 'abigail', null])->map(function (string $name) {
        return strtoupper($name);
    })->reject(function (string $name) {
        return empty($name);
    });

Как видите, `Collection` класс позволяет вам связывать свои методы для выполнения плавного отображения и сокращения базового массива. Как правило, коллекции неизменяемы, то есть каждый `Collection` метод возвращает совершенно новый `Collection` экземпляр.

<a name="creating-collections"></a>
### Создание коллекций

Как упоминалось выше, `collect` помощник возвращает новый `Mnv\Core\Collections\Collection` экземпляр для данного массива. Итак, создать коллекцию так же просто, как:

    $collection = collect([1, 2, 3]);

> **Примечание**  
> Результаты запросов всегда возвращаются в виде `Collection` экземпляров.

<a name="available-methods"></a>
## Доступные методы

В большей части оставшейся документации коллекции мы обсудим каждый метод, доступный в классе `Collection`. Помните, что все эти методы могут быть объединены в цепочку для плавного управления базовым массивом. 
Кроме того, почти каждый метод возвращает новый `Collection` экземпляр, что позволяет при необходимости сохранить исходную копию коллекции:


[all](#method-all)
[average](#method-average)
[avg](#method-avg)
[chunk](#method-chunk)
[chunkWhile](#method-chunkwhile)
[collapse](#method-collapse)
[collect](#method-collect)
[combine](#method-combine)
[concat](#method-concat)
[contains](#method-contains)
[containsOneItem](#method-containsoneitem)
[containsStrict](#method-containsstrict)
[count](#method-count)
[countBy](#method-countBy)
[crossJoin](#method-crossjoin)
[diff](#method-diff)
[diffAssoc](#method-diffassoc)
[diffKeys](#method-diffkeys)
[doesntContain](#method-doesntcontain)
[duplicates](#method-duplicates)
[duplicatesStrict](#method-duplicatesstrict)
[each](#method-each)
[eachSpread](#method-eachspread)
[every](#method-every)
[except](#method-except)
[filter](#method-filter)
[first](#method-first)
[firstOrFail](#method-first-or-fail)
[firstWhere](#method-first-where)
[flatMap](#method-flatmap)
[flatten](#method-flatten)
[flip](#method-flip)
[forget](#method-forget)
[forPage](#method-forpage)
[get](#method-get)
[groupBy](#method-groupby)
[has](#method-has)
[hasAny](#method-hasany)
[implode](#method-implode)
[intersect](#method-intersect)
[intersectAssoc](#method-intersectAssoc)
[intersectByKeys](#method-intersectbykeys)
[isEmpty](#method-isempty)
[isNotEmpty](#method-isnotempty)
[join](#method-join)
[keyBy](#method-keyby)
[keys](#method-keys)
[last](#method-last)
[lazy](#method-lazy)
[macro](#method-macro)
[make](#method-make)
[map](#method-map)
[mapInto](#method-mapinto)
[mapSpread](#method-mapspread)
[mapToGroups](#method-maptogroups)
[mapWithKeys](#method-mapwithkeys)
[max](#method-max)
[median](#method-median)
[merge](#method-merge)
[mergeRecursive](#method-mergerecursive)
[min](#method-min)
[mode](#method-mode)
[nth](#method-nth)
[only](#method-only)
[pad](#method-pad)
[partition](#method-partition)
[pipe](#method-pipe)
[pipeInto](#method-pipeinto)
[pipeThrough](#method-pipethrough)
[pluck](#method-pluck)
[pop](#method-pop)
[prepend](#method-prepend)
[pull](#method-pull)
[push](#method-push)
[put](#method-put)
[random](#method-random)
[range](#method-range)
[reduce](#method-reduce)
[reduceSpread](#method-reduce-spread)
[reject](#method-reject)
[replace](#method-replace)
[replaceRecursive](#method-replacerecursive)
[reverse](#method-reverse)
[search](#method-search)
[shift](#method-shift)
[shuffle](#method-shuffle)
[skip](#method-skip)
[skipUntil](#method-skipuntil)
[skipWhile](#method-skipwhile)
[slice](#method-slice)
[sliding](#method-sliding)
[sole](#method-sole)
[some](#method-some)
[sort](#method-sort)
[sortBy](#method-sortby)
[sortByDesc](#method-sortbydesc)
[sortDesc](#method-sortdesc)
[sortKeys](#method-sortkeys)
[sortKeysDesc](#method-sortkeysdesc)
[sortKeysUsing](#method-sortkeysusing)
[splice](#method-splice)
[split](#method-split)
[splitIn](#method-splitin)
[sum](#method-sum)
[take](#method-take)
[takeUntil](#method-takeuntil)
[takeWhile](#method-takewhile)
[tap](#method-tap)
[times](#method-times)
[toArray](#method-toarray)
[toJson](#method-tojson)
[transform](#method-transform)
[undot](#method-undot)
[union](#method-union)
[unique](#method-unique)
[uniqueStrict](#method-uniquestrict)
[unless](#method-unless)
[unlessEmpty](#method-unlessempty)
[unlessNotEmpty](#method-unlessnotempty)
[unwrap](#method-unwrap)
[value](#method-value)
[values](#method-values)
[when](#method-when)
[whenEmpty](#method-whenempty)
[whenNotEmpty](#method-whennotempty)
[where](#method-where)
[whereStrict](#method-wherestrict)
[whereBetween](#method-wherebetween)
[whereIn](#method-wherein)
[whereInStrict](#method-whereinstrict)
[whereInstanceOf](#method-whereinstanceof)
[whereNotBetween](#method-wherenotbetween)
[whereNotIn](#method-wherenotin)
[whereNotInStrict](#method-wherenotinstrict)
[whereNotNull](#method-wherenotnull)
[whereNull](#method-wherenull)
[wrap](#method-wrap)
[zip](#method-zip)


<a name="method-listing"></a>
## Список методов

<a name="method-all"></a>
#### `all()`

Метод `all` возвращает базовый массив, представленный коллекцией:

    collect([1, 2, 3])->all();

    // [1, 2, 3]

<a name="method-average"></a>
#### `average()` 

Псевдоним метода [`avg`](#method-avg).

<a name="method-avg"></a>
#### `avg()`

Метод `avg` возвращает среднее значение данного ключа:

    $average = collect([
        ['foo' => 10],
        ['foo' => 10],
        ['foo' => 20],
        ['foo' => 40]
    ])->avg('foo');

    // 20

    $average = collect([1, 1, 2, 4])->avg();

    // 2

<a name="method-chunk"></a>
#### `chunk()`

Метод `chunk` разбивает коллекцию на несколько меньших коллекций заданного размера:

    $collection = collect([1, 2, 3, 4, 5, 6, 7]);

    $chunks = $collection->chunk(4);

    $chunks->all();

    // [[1, 2, 3, 4], [5, 6, 7]]

Этот метод особенно полезен в представлениях при работе с сеточной системой, такой как `Bootstrap` . 
Например, представьте, что у вас есть коллекция моделей, которые вы хотите отобразить в виде сетки:

```blade
@foreach ($products->chunk(3) as $chunk)
    <div class="row">
        @foreach ($chunk as $product)
            <div class="col-xs-4">{{ $product->name }}</div>
        @endforeach
    </div>
@endforeach
```

<a name="method-chunkwhile"></a>
#### `chunkWhile()`

Метод `chunkWhile` разбивает коллекцию на несколько меньших коллекций на основе оценки данного обратного вызова. 
Переменная `$chunk`, переданная замыканию, может использоваться для проверки предыдущего элемента:

    $collection = collect(str_split('AABBCCCD'));

    $chunks = $collection->chunkWhile(function (string $value, int $key, Collection $chunk) {
        return $value === $chunk->last();
    });

    $chunks->all();

    // [['A', 'A'], ['B', 'B'], ['C', 'C', 'C'], ['D']]

<a name="method-collapse"></a>
#### `collapse()`

Метод `collapse` сворачивает коллекцию массивов в одну плоскую коллекцию:

    $collection = collect([
        [1, 2, 3],
        [4, 5, 6],
        [7, 8, 9],
    ]);

    $collapsed = $collection->collapse();

    $collapsed->all();

    // [1, 2, 3, 4, 5, 6, 7, 8, 9]

<a name="method-collect"></a>
#### `collect()`

Метод `collect` возвращает новый `Collection` экземпляр с элементами, которые в настоящее время находятся в коллекции:

    $collectionA = collect([1, 2, 3]);

    $collectionB = $collectionA->collect();

    $collectionB->all();

    // [1, 2, 3]

Этот `collect` метод в первую очередь полезен для преобразования [lazy collections](#lazy-collections) в стандартные `Collection` экземпляры:

    $lazyCollection = LazyCollection::make(function () {
        yield 1;
        yield 2;
        yield 3;
    });

    $collection = $lazyCollection->collect();

    get_class($collection);

    // 'Mnv\Core\Collections\Collection'

    $collection->all();

    // [1, 2, 3]

> **Примечание**  
> Этот `collect` метод особенно полезен, когда у вас есть экземпляр `Enumerable` неленивого экземпляра коллекции, и вам нужен экземпляр. Поскольку `collect()` это часть контракта Enumerable, вы можете смело использовать его для получения экземпляра `Collection`.

<a name="method-combine"></a>
#### `combine()`

Метод `combine` объединяет значения коллекции в качестве ключей со значениями другого массива или коллекции:

    $collection = collect(['name', 'age']);

    $combined = $collection->combine(['George', 29]);

    $combined->all();

    // ['name' => 'George', 'age' => 29]

<a name="method-concat"></a>
#### `concat()`

Метод `concat` добавляет данные `array` или значения коллекции в конец другой коллекции:

    $collection = collect(['John Doe']);

    $concatenated = $collection->concat(['Jane Doe'])->concat(['name' => 'Johnny Doe']);

    $concatenated->all();

    // ['John Doe', 'Jane Doe', 'Johnny Doe']

Метод `concat` численно переиндексирует ключи для элементов, объединенных в исходную коллекцию. Чтобы сохранить ключи в ассоциативных коллекциях, см. метод [merge](#method-merge).

<a name="method-contains"></a>
#### `contains()`

Метод `contains` определяет, содержит ли коллекция данный элемент. 
Вы можете передать методу замыкание `contains`, чтобы определить, существует ли в коллекции элемент, соответствующий данному критерию истинности:

    $collection = collect([1, 2, 3, 4, 5]);

    $collection->contains(function (int $value, int $key) {
        return $value > 5;
    });

    // false

В качестве альтернативы вы можете передать строку методу `contains`, чтобы определить, содержит ли коллекция заданное значение элемента:

    $collection = collect(['name' => 'Desk', 'price' => 100]);

    $collection->contains('Desk');

    // true

    $collection->contains('New York');

    // false

Вы также можете передать пару ключ/значение методу `contains`, который определит, существует ли данная пара в коллекции:

    $collection = collect([
        ['product' => 'Desk', 'price' => 200],
        ['product' => 'Chair', 'price' => 100],
    ]);

    $collection->contains('product', 'Bookcase');

    // false

Метод `contains` использует «свободные» сравнения при проверке значений элементов, то есть строка с целочисленным значением будет считаться равной целому числу с тем же значением. Используйте [`containsStrict`](#method-containsstrict) метод для фильтрации с использованием «строгих» сравнений.

Обратное см. `contains` в методе [doesntContain](#method-doesntcontain).

<a name="method-containsoneitem"></a>
#### `containsOneItem()`

Метод `containsOneItem` определяет, содержит ли коллекция один элемент:

    collect([])->containsOneItem();

    // false

    collect(['1'])->containsOneItem();

    // true

    collect(['1', '2'])->containsOneItem();

    // false

<a name="method-containsstrict"></a>
#### `containsStrict()`

Этот метод имеет ту же сигнатуру, что и [`contains`](#method-contains) метод; однако все значения сравниваются с использованием «строгих» сравнений.

> **Примечание**  
> Поведение этого метода изменяется при использовании  [Eloquent Collections](/docs/eloquent-collections#method-contains).

<a name="method-count"></a>
#### `count()`

Метод `count` возвращает общее количество элементов в коллекции:

    $collection = collect([1, 2, 3, 4]);

    $collection->count();

    // 4

<a name="method-countBy"></a>
#### `countBy()`

Метод `countBy` подсчитывает вхождения значений в коллекцию. По умолчанию метод подсчитывает количество вхождений каждого элемента, что позволяет вам подсчитывать определенные «типы» элементов в коллекции:

    $collection = collect([1, 2, 2, 2, 3]);

    $counted = $collection->countBy();

    $counted->all();

    // [1 => 1, 2 => 3, 3 => 1]

Вы передаете замыкание методу `countBy` для подсчета всех элементов по пользовательскому значению:

    $collection = collect(['alice@gmail.com', 'bob@yahoo.com', 'carlos@gmail.com']);

    $counted = $collection->countBy(function (string $email) {
        return substr(strrchr($email, "@"), 1);
    });

    $counted->all();

    // ['gmail.com' => 2, 'yahoo.com' => 1]

<a name="method-crossjoin"></a>
#### `crossJoin()`

Метод `crossJoin` cross объединяет значения коллекции среди заданных массивов или коллекций, возвращая декартово произведение со всеми возможными перестановками:

    $collection = collect([1, 2]);

    $matrix = $collection->crossJoin(['a', 'b']);

    $matrix->all();

    /*
        [
            [1, 'a'],
            [1, 'b'],
            [2, 'a'],
            [2, 'b'],
        ]
    */

    $collection = collect([1, 2]);

    $matrix = $collection->crossJoin(['a', 'b'], ['I', 'II']);

    $matrix->all();

    /*
        [
            [1, 'a', 'I'],
            [1, 'a', 'II'],
            [1, 'b', 'I'],
            [1, 'b', 'II'],
            [2, 'a', 'I'],
            [2, 'a', 'II'],
            [2, 'b', 'I'],
            [2, 'b', 'II'],
        ]
    */

<a name="method-diff"></a>
#### `diff()` {.collection-method}

Метод `diff` сравнивает коллекцию с другой коллекцией или простым PHP `array` на основе ее значений. Этот метод вернет значения в исходной коллекции, которых нет в данной коллекции:

    $collection = collect([1, 2, 3, 4, 5]);

    $diff = $collection->diff([2, 4, 6, 8]);

    $diff->all();

    // [1, 3, 5]

<a name="method-diffassoc"></a>
#### `diffAssoc()`

Метод `diffAssoc` сравнивает коллекцию с другой коллекцией или простым PHP `array` на основе ее ключей и значений. Этот метод вернет пары ключ/значение в исходной коллекции, которых нет в данной коллекции:

    $collection = collect([
        'color' => 'orange',
        'type' => 'fruit',
        'remain' => 6,
    ]);

    $diff = $collection->diffAssoc([
        'color' => 'yellow',
        'type' => 'fruit',
        'remain' => 3,
        'used' => 6,
    ]);

    $diff->all();

    // ['color' => 'orange', 'remain' => 6]

<a name="method-diffkeys"></a>
#### `diffKeys()`

Метод `diffKeys` сравнивает коллекцию с другой коллекцией или простым PHP `array` на основе ее ключей. Этот метод вернет пары ключ/значение в исходной коллекции, которых нет в данной коллекции:

    $collection = collect([
        'one' => 10,
        'two' => 20,
        'three' => 30,
        'four' => 40,
        'five' => 50,
    ]);

    $diff = $collection->diffKeys([
        'two' => 2,
        'four' => 4,
        'six' => 6,
        'eight' => 8,
    ]);

    $diff->all();

    // ['one' => 10, 'three' => 30, 'five' => 50]

<a name="method-doesntcontain"></a>
#### `doesntContain()`

Метод `doesntContain` определяет, содержит ли коллекция заданный элемент. Вы можете передать методу замыкание `doesntContain`, чтобы определить, не существует ли в коллекции элемента, соответствующего заданному тесту на истинность:

    $collection = collect([1, 2, 3, 4, 5]);

    $collection->doesntContain(function (int $value, int $key) {
        return $value < 5;
    });

    // false

В качестве альтернативы вы можете передать строку методу `doesntContain`, чтобы определить, не содержит ли коллекция заданное значение элемента:

    $collection = collect(['name' => 'Desk', 'price' => 100]);

    $collection->doesntContain('Table');

    // true

    $collection->doesntContain('Desk');

    // false

Вы также можете передать пару ключ/значение методу `doesntContain`, который определит, не существует ли данная пара в коллекции:

    $collection = collect([
        ['product' => 'Desk', 'price' => 200],
        ['product' => 'Chair', 'price' => 100],
    ]);

    $collection->doesntContain('product', 'Bookcase');

    // true

Метод `doesntContain` использует «свободные» сравнения при проверке значений элементов, то есть строка с целочисленным значением будет считаться равной целому числу с тем же значением.

<a name="method-duplicates"></a>
#### `duplicates()`

Метод `duplicates` извлекает и возвращает повторяющиеся значения из коллекции:

    $collection = collect(['a', 'b', 'a', 'c', 'b']);

    $collection->duplicates();

    // [2 => 'a', 4 => 'b']

Если коллекция содержит массивы или объекты, вы можете передать ключ атрибутов, которые вы хотите проверить на повторяющиеся значения:

    $employees = collect([
        ['email' => 'abigail@example.com', 'position' => 'Developer'],
        ['email' => 'james@example.com', 'position' => 'Designer'],
        ['email' => 'victoria@example.com', 'position' => 'Developer'],
    ]);

    $employees->duplicates('position');

    // [2 => 'Developer']

<a name="method-duplicatesstrict"></a>
#### `duplicatesStrict()`

Этот метод имеет ту же сигнатуру, что и [`duplicates`](#method-duplicates) метод; однако все значения сравниваются с использованием «строгих» сравнений.

<a name="method-each"></a>
#### `each()`

Метод `each` перебирает элементы в коллекции и передает каждый элемент в замыкание:

    $collection = collect([1, 2, 3, 4]);

    $collection->each(function (int $item, int $key) {
        // ...
    });

Если вы хотите прекратить повторение элементов, вы можете вернуться `false` из закрытия:

    $collection->each(function (int $item, int $key) {
        if (/* condition */) {
            return false;
        }
    });

<a name="method-eachspread"></a>
#### `eachSpread()`

Метод `eachSpread` перебирает элементы коллекции, передавая значение каждого вложенного элемента в заданный обратный вызов:

    $collection = collect([['John Doe', 35], ['Jane Doe', 33]]);

    $collection->eachSpread(function (string $name, int $age) {
        // ...
    });

Вы можете остановить перебор элементов, вернувшись `false` из обратного вызова:

    $collection->eachSpread(function (string $name, int $age) {
        return false;
    });

<a name="method-every"></a>
#### `every()`

Этот `every` метод можно использовать для проверки того, что все элементы коллекции проходят заданный тест на истинность:

    collect([1, 2, 3, 4])->every(function (int $value, int $key) {
        return $value > 2;
    });

    // false

Если коллекция пуста, `every` метод вернет true:

    $collection = collect([]);

    $collection->every(function (int $value, int $key) {
        return $value > 2;
    });

    // true

<a name="method-except"></a>
#### `except()`

Метод `except` возвращает все элементы коллекции, кроме тех, у которых есть указанные ключи:

    $collection = collect(['product_id' => 1, 'price' => 100, 'discount' => false]);

    $filtered = $collection->except(['price', 'discount']);

    $filtered->all();

    // ['product_id' => 1]

Для инверсии `except` см. [only](#method-only) метод.

<a name="method-filter"></a>
#### `filter()`

Метод `filter` фильтрует коллекцию, используя заданный обратный вызов, оставляя только те элементы, которые проходят заданный тест на истинность:

    $collection = collect([1, 2, 3, 4]);

    $filtered = $collection->filter(function (int $value, int $key) {
        return $value > 2;
    });

    $filtered->all();

    // [3, 4]

`false` Если обратный вызов не указан, будут удалены все записи коллекции, эквивалентные :

    $collection = collect([1, 2, 3, null, false, '', 0, []]);

    $collection->filter()->all();

    // [1, 2, 3]

Обратное см. `filter` в методе [reject](#method-reject).

<a name="method-first"></a>
#### `first()`

Метод `first` возвращает первый элемент в коллекции, который проходит заданный тест на истинность:

    collect([1, 2, 3, 4])->first(function (int $value, int $key) {
        return $value > 2;
    });

    // 3

Вы также можете вызвать `first` метод без аргументов, чтобы получить первый элемент в коллекции. Если коллекция пуста, `null` возвращается:

    collect([1, 2, 3, 4])->first();

    // 1

<a name="method-first-or-fail"></a>
#### `firstOrFail()`

Метод `firstOrFail` идентичен методу `first`; однако, если результат не найден, `Mnv\Core\Collections\ItemNotFoundException` будет выдано исключение:

    collect([1, 2, 3, 4])->firstOrFail(function (int $value, int $key) {
        return $value > 5;
    });

    // Throws ItemNotFoundException...

Вы также можете вызвать `firstOrFail` метод без аргументов, чтобы получить первый элемент в коллекции. Если коллекция пуста, `Mnv\Core\Collections\ItemNotFoundException` будет выброшено исключение:

    collect([])->firstOrFail();

    // Throws ItemNotFoundException...

<a name="method-first-where"></a>
#### `firstWhere()`

Метод `firstWhere` возвращает первый элемент коллекции с заданной парой ключ/значение:

    $collection = collect([
        ['name' => 'Regena', 'age' => null],
        ['name' => 'Linda', 'age' => 14],
        ['name' => 'Diego', 'age' => 23],
        ['name' => 'Linda', 'age' => 84],
    ]);

    $collection->firstWhere('name', 'Linda');

    // ['name' => 'Linda', 'age' => 14]

Вы также можете вызвать `firstWhere` метод с оператором сравнения:

    $collection->firstWhere('age', '>=', 18);

    // ['name' => 'Diego', 'age' => 23]

Как и в случае с методом [where](#method-where)  , вы можете передать методу один аргумент `firstWhere`. В этом сценарии `firstWhere` метод вернет первый элемент, где значение данного ключа элемента является «истинным»:

    $collection->firstWhere('age');

    // ['name' => 'Linda', 'age' => 14]

<a name="method-flatmap"></a>
#### `flatMap()`

Метод `flatMap` выполняет итерацию по коллекции и передает каждое значение данному замыканию. 
Закрытие может изменять элемент и возвращать его, таким образом формируя новую коллекцию измененных элементов. Затем массив выравнивается на один уровень:

    $collection = collect([
        ['name' => 'Sally'],
        ['school' => 'Arkansas'],
        ['age' => 28]
    ]);

    $flattened = $collection->flatMap(function (array $values) {
        return array_map('strtoupper', $values);
    });

    $flattened->all();

    // ['name' => 'SALLY', 'school' => 'ARKANSAS', 'age' => '28'];

<a name="method-flatten"></a>
#### `flatten()` 

Метод `flatten` сводит многомерную коллекцию в одно измерение:

    $collection = collect([
        'name' => 'taylor',
        'languages' => [
            'php', 'javascript'
        ]
    ]);

    $flattened = $collection->flatten();

    $flattened->all();

    // ['taylor', 'php', 'javascript'];

При необходимости вы можете передать `flatten` методу аргумент «глубина»:

    $collection = collect([
        'Apple' => [
            [
                'name' => 'iPhone 6S',
                'brand' => 'Apple'
            ],
        ],
        'Samsung' => [
            [
                'name' => 'Galaxy S7',
                'brand' => 'Samsung'
            ],
        ],
    ]);

    $products = $collection->flatten(1);

    $products->values()->all();

    /*
        [
            ['name' => 'iPhone 6S', 'brand' => 'Apple'],
            ['name' => 'Galaxy S7', 'brand' => 'Samsung'],
        ]
    */

В этом примере вызов `flatten` без указания глубины также сгладил бы вложенные массивы, что привело бы к ['iPhone 6S', 'Apple', 'Galaxy S7', 'Samsung']. Предоставление глубины позволяет вам указать количество уровней вложенных массивов, которые будут сведены.

<a name="method-flip"></a>
#### `flip()` 

Метод `flip` меняет местами ключи коллекции с их соответствующими значениями:

    $collection = collect(['name' => 'taylor', 'framework' => 'laravel']);

    $flipped = $collection->flip();

    $flipped->all();

    // ['taylor' => 'name', 'laravel' => 'framework']

<a name="method-forget"></a>
#### `forget()`

Метод `forget` удаляет элемент из коллекции по его ключу:

    $collection = collect(['name' => 'taylor', 'framework' => 'laravel']);

    $collection->forget('name');

    $collection->all();

    // ['framework' => 'laravel']

> **Предупреждение**  
> В отличие от большинства других методов сбора, `forget` не возвращает новую измененную коллекцию; он изменяет вызываемую коллекцию.

<a name="method-forpage"></a>
#### `forPage()`

Метод `forPage` возвращает новую коллекцию, содержащую элементы, которые должны присутствовать на странице с заданным номером. Метод принимает номер страницы в качестве первого аргумента и количество элементов, отображаемых на странице, в качестве второго аргумента:

    $collection = collect([1, 2, 3, 4, 5, 6, 7, 8, 9]);

    $chunk = $collection->forPage(2, 3);

    $chunk->all();

    // [4, 5, 6]

<a name="method-get"></a>
#### `get()` 

Метод `get` возвращает элемент по заданному ключу. Если ключ не существует, `null` возвращается:

    $collection = collect(['name' => 'taylor', 'framework' => 'laravel']);

    $value = $collection->get('name');

    // taylor

При желании вы можете передать значение по умолчанию в качестве второго аргумента:

    $collection = collect(['name' => 'taylor', 'framework' => 'laravel']);

    $value = $collection->get('age', 34);

    // 34

Вы даже можете передать обратный вызов в качестве значения метода по умолчанию. Результат обратного вызова будет возвращен, если указанный ключ не существует:

    $collection->get('email', function () {
        return 'taylor@example.com';
    });

    // taylor@example.com

<a name="method-groupby"></a>
#### `groupBy()` 

Метод `groupBy` группирует элементы коллекции по заданному ключу:

    $collection = collect([
        ['account_id' => 'account-x10', 'product' => 'Chair'],
        ['account_id' => 'account-x10', 'product' => 'Bookcase'],
        ['account_id' => 'account-x11', 'product' => 'Desk'],
    ]);

    $grouped = $collection->groupBy('account_id');

    $grouped->all();

    /*
        [
            'account-x10' => [
                ['account_id' => 'account-x10', 'product' => 'Chair'],
                ['account_id' => 'account-x10', 'product' => 'Bookcase'],
            ],
            'account-x11' => [
                ['account_id' => 'account-x11', 'product' => 'Desk'],
            ],
        ]
    */

Вместо передачи строки `key` вы можете передать обратный вызов. Обратный вызов должен возвращать значение, которое вы хотите ввести для группы:

    $grouped = $collection->groupBy(function (array $item, int $key) {
        return substr($item['account_id'], -3);
    });

    $grouped->all();

    /*
        [
            'x10' => [
                ['account_id' => 'account-x10', 'product' => 'Chair'],
                ['account_id' => 'account-x10', 'product' => 'Bookcase'],
            ],
            'x11' => [
                ['account_id' => 'account-x11', 'product' => 'Desk'],
            ],
        ]
    */

Несколько критериев группировки могут быть переданы в виде массива. Каждый элемент массива будет применяться к соответствующему уровню в многомерном массиве:

    $data = new Collection([
        10 => ['user' => 1, 'skill' => 1, 'roles' => ['Role_1', 'Role_3']],
        20 => ['user' => 2, 'skill' => 1, 'roles' => ['Role_1', 'Role_2']],
        30 => ['user' => 3, 'skill' => 2, 'roles' => ['Role_1']],
        40 => ['user' => 4, 'skill' => 2, 'roles' => ['Role_2']],
    ]);

    $result = $data->groupBy(['skill', function (array $item) {
        return $item['roles'];
    }], preserveKeys: true);

    /*
    [
        1 => [
            'Role_1' => [
                10 => ['user' => 1, 'skill' => 1, 'roles' => ['Role_1', 'Role_3']],
                20 => ['user' => 2, 'skill' => 1, 'roles' => ['Role_1', 'Role_2']],
            ],
            'Role_2' => [
                20 => ['user' => 2, 'skill' => 1, 'roles' => ['Role_1', 'Role_2']],
            ],
            'Role_3' => [
                10 => ['user' => 1, 'skill' => 1, 'roles' => ['Role_1', 'Role_3']],
            ],
        ],
        2 => [
            'Role_1' => [
                30 => ['user' => 3, 'skill' => 2, 'roles' => ['Role_1']],
            ],
            'Role_2' => [
                40 => ['user' => 4, 'skill' => 2, 'roles' => ['Role_2']],
            ],
        ],
    ];
    */

<a name="method-has"></a>
#### `has()` 

Метод `has` определяет, существует ли данный ключ в коллекции:

    $collection = collect(['account_id' => 1, 'product' => 'Desk', 'amount' => 5]);

    $collection->has('product');

    // true

    $collection->has(['product', 'amount']);

    // true

    $collection->has(['amount', 'price']);

    // false

<a name="method-hasany"></a>
#### `hasAny()` 

Метод `hasAny` определяет, существует ли какой-либо из заданных ключей в коллекции:

    $collection = collect(['account_id' => 1, 'product' => 'Desk', 'amount' => 5]);

    $collection->hasAny(['product', 'price']);

    // true

    $collection->hasAny(['name', 'price']);

    // false

<a name="method-implode"></a>
#### `implode()` 

Метод `implode` объединяет элементы в коллекцию. Его аргументы зависят от типа элементов в коллекции. Если коллекция содержит массивы или объекты, вы должны передать ключ атрибутов, которые вы хотите соединить, и строку «склеивания», которую вы хотите поместить между значениями:

    $collection = collect([
        ['account_id' => 1, 'product' => 'Desk'],
        ['account_id' => 2, 'product' => 'Chair'],
    ]);

    $collection->implode('product', ', ');

    // Desk, Chair

Если коллекция содержит простые строки или числовые значения, вы должны передать «клей» в качестве единственного аргумента метода:

    collect([1, 2, 3, 4, 5])->implode('-');

    // '1-2-3-4-5'

Вы можете передать методу замыкание, `implode` если хотите отформатировать значения, которые будут сжаты:

    $collection->implode(function (array $item, int $key) {
        return strtoupper($item['product']);
    }, ', ');

    // DESK, CHAIR

<a name="method-intersect"></a>
#### `intersect()`

Метод `intersect` удаляет все значения из исходной коллекции, которых нет в заданной arrayили коллекции. Полученная коллекция сохранит ключи исходной коллекции:

    $collection = collect(['Desk', 'Sofa', 'Chair']);

    $intersect = $collection->intersect(['Desk', 'Chair', 'Bookcase']);

    $intersect->all();

    // [0 => 'Desk', 2 => 'Chair']

<a name="method-intersectAssoc"></a>
#### `intersectAssoc()`

Метод `intersectAssoc` сравнивает исходную коллекцию с другой коллекцией или array, возвращая пары ключ/значение, присутствующие во всех заданных коллекциях:

    $collection = collect([
        'color' => 'red',
        'size' => 'M',
        'material' => 'cotton'
    ]);

    $intersect = $collection->intersectAssoc([
        'color' => 'blue',
        'size' => 'M',
        'material' => 'polyester'
    ]);

    $intersect->all();

    // ['size' => 'M']

<a name="method-intersectbykeys"></a>
#### `intersectByKeys()` 

Метод `intersectByKeys` удаляет любые ключи и соответствующие им значения из исходной коллекции, которых нет в заданной `array` или коллекции:

    $collection = collect([
        'serial' => 'UX301', 'type' => 'screen', 'year' => 2009,
    ]);

    $intersect = $collection->intersectByKeys([
        'reference' => 'UX404', 'type' => 'tab', 'year' => 2011,
    ]);

    $intersect->all();

    // ['type' => 'screen', 'year' => 2009]

<a name="method-isempty"></a>
#### `isEmpty()`

Метод `isEmpty` возвращает значение `true`, если коллекция пуста; в противном случае `false` возвращается:

    collect([])->isEmpty();

    // true

<a name="method-isnotempty"></a>
#### `isNotEmpty()`

Метод `isNotEmpty` возвращает значение `true`, если коллекция не пуста; в противном случае `false` возвращается:

    collect([])->isNotEmpty();

    // false

<a name="method-join"></a>
#### `join()`

Метод `join` объединяет значения коллекции строкой. Используя второй аргумент этого метода, вы также можете указать, как последний элемент должен быть добавлен к строке:

    collect(['a', 'b', 'c'])->join(', '); // 'a, b, c'
    collect(['a', 'b', 'c'])->join(', ', ', and '); // 'a, b, and c'
    collect(['a', 'b'])->join(', ', ' and '); // 'a and b'
    collect(['a'])->join(', ', ' and '); // 'a'
    collect([])->join(', ', ' and '); // ''

<a name="method-keyby"></a>
#### `keyBy()` 

Метод `keyBy` вводит коллекцию по заданному ключу. Если несколько элементов имеют один и тот же ключ, в новой коллекции появится только последний:

    $collection = collect([
        ['product_id' => 'prod-100', 'name' => 'Desk'],
        ['product_id' => 'prod-200', 'name' => 'Chair'],
    ]);

    $keyed = $collection->keyBy('product_id');

    $keyed->all();

    /*
        [
            'prod-100' => ['product_id' => 'prod-100', 'name' => 'Desk'],
            'prod-200' => ['product_id' => 'prod-200', 'name' => 'Chair'],
        ]
    */

Вы также можете передать обратный вызов методу. Обратный вызов должен возвращать значение для ключа коллекции:

    $keyed = $collection->keyBy(function (array $item, int $key) {
        return strtoupper($item['product_id']);
    });

    $keyed->all();

    /*
        [
            'PROD-100' => ['product_id' => 'prod-100', 'name' => 'Desk'],
            'PROD-200' => ['product_id' => 'prod-200', 'name' => 'Chair'],
        ]
    */

<a name="method-keys"></a>
#### `keys()`

Метод `keys` возвращает все ключи коллекции:

    $collection = collect([
        'prod-100' => ['product_id' => 'prod-100', 'name' => 'Desk'],
        'prod-200' => ['product_id' => 'prod-200', 'name' => 'Chair'],
    ]);

    $keys = $collection->keys();

    $keys->all();

    // ['prod-100', 'prod-200']

<a name="method-last"></a>
#### `last()`

Метод `last` возвращает последний элемент в коллекции, который проходит заданный тест на истинность:

    collect([1, 2, 3, 4])->last(function (int $value, int $key) {
        return $value < 3;
    });

    // 2

Вы также можете вызвать `last` метод без аргументов, чтобы получить последний элемент в коллекции. Если коллекция пуста, `null` возвращается:

    collect([1, 2, 3, 4])->last();

    // 4

<a name="method-lazy"></a>
#### `lazy()`

Метод `lazy` возвращает новый [`LazyCollection`](#lazy-collections) экземпляр из базового массива элементов:

    $lazyCollection = collect([1, 2, 3, 4])->lazy();

    get_class($lazyCollection);

    // Mnv\Core\Collections\LazyCollection

    $lazyCollection->all();

    // [1, 2, 3, 4]

Это особенно полезно, когда вам нужно выполнить преобразования на огромном `Collection`, содержащем много элементов:

    $count = $hugeCollection
        ->lazy()
        ->where('country', 'FR')
        ->where('balance', '>', '100')
        ->count();

Преобразовывая коллекцию в `LazyCollection`, мы избегаем выделения тонны дополнительной памяти. Хотя исходная коллекция по-прежнему хранит свои значения в памяти, последующие фильтры этого не сделают. Поэтому при фильтрации результатов коллекции практически не выделяется дополнительная память.

<a name="method-macro"></a>
#### `macro()` 

Статический `macro` метод позволяет добавлять методы в `Collection` класс во время выполнения. Обратитесь к документации по [extending collections](#extending-collections) для получения дополнительной информации.

<a name="method-make"></a>
#### `make()`

Статический `make` метод создает новый экземпляр коллекции. См. раздел [Creating Collections](#creating-collections).

<a name="method-map"></a>
#### `map()`

Метод `map` перебирает коллекцию и передает каждое значение заданному обратному вызову. Обратный вызов может изменять элемент и возвращать его, таким образом формируя новую коллекцию измененных элементов:

    $collection = collect([1, 2, 3, 4, 5]);

    $multiplied = $collection->map(function (int $item, int $key) {
        return $item * 2;
    });

    $multiplied->all();

    // [2, 4, 6, 8, 10]

> **Предупреждение**  
> Как и большинство других методов коллекции, `map` возвращает новый экземпляр коллекции; он не изменяет вызываемую коллекцию. Если вы хотите преобразовать исходную коллекцию, используйте [`transform`](#method-transform).

<a name="method-mapinto"></a>
#### `mapInto()`

Метод `mapInto()` перебирает коллекцию, создавая новый экземпляр данного класса, передавая значение в конструктор:

    class Currency
    {
        /**
         * Create a new currency instance.
         */
        function __construct(
            public string $code
        ) {}
    }

    $collection = collect(['USD', 'EUR', 'GBP']);

    $currencies = $collection->mapInto(Currency::class);

    $currencies->all();

    // [Currency('USD'), Currency('EUR'), Currency('GBP')]

<a name="method-mapspread"></a>
#### `mapSpread()`

Метод `mapSpread` выполняет итерацию по элементам коллекции, передавая значение каждого вложенного элемента в данное замыкание. Закрытие может изменять элемент и возвращать его, таким образом формируя новую коллекцию измененных элементов:

    $collection = collect([0, 1, 2, 3, 4, 5, 6, 7, 8, 9]);

    $chunks = $collection->chunk(2);

    $sequence = $chunks->mapSpread(function (int $even, int $odd) {
        return $even + $odd;
    });

    $sequence->all();

    // [1, 5, 9, 13, 17]

<a name="method-maptogroups"></a>
#### `mapToGroups()`

Метод `mapToGroups` группирует элементы коллекции по заданному замыканию. Замыкание должно возвращать ассоциативный массив, содержащий одну пару ключ/значение, таким образом формируя новую коллекцию сгруппированных значений:

    $collection = collect([
        [
            'name' => 'John Doe',
            'department' => 'Sales',
        ],
        [
            'name' => 'Jane Doe',
            'department' => 'Sales',
        ],
        [
            'name' => 'Johnny Doe',
            'department' => 'Marketing',
        ]
    ]);

    $grouped = $collection->mapToGroups(function (array $item, int $key) {
        return [$item['department'] => $item['name']];
    });

    $grouped->all();

    /*
        [
            'Sales' => ['John Doe', 'Jane Doe'],
            'Marketing' => ['Johnny Doe'],
        ]
    */

    $grouped->get('Sales')->all();

    // ['John Doe', 'Jane Doe']

<a name="method-mapwithkeys"></a>
#### `mapWithKeys()`

Метод `mapWithKeys` перебирает коллекцию и передает каждое значение заданному обратному вызову. Обратный вызов должен возвращать ассоциативный массив, содержащий одну пару ключ/значение:

    $collection = collect([
        [
            'name' => 'John',
            'department' => 'Sales',
            'email' => 'john@example.com',
        ],
        [
            'name' => 'Jane',
            'department' => 'Marketing',
            'email' => 'jane@example.com',
        ]
    ]);

    $keyed = $collection->mapWithKeys(function (array $item, int $key) {
        return [$item['email'] => $item['name']];
    });

    $keyed->all();

    /*
        [
            'john@example.com' => 'John',
            'jane@example.com' => 'Jane',
        ]
    */

<a name="method-max"></a>
#### `max()`

Метод `max` возвращает максимальное значение данного ключа:

    $max = collect([
        ['foo' => 10],
        ['foo' => 20]
    ])->max('foo');

    // 20

    $max = collect([1, 2, 3, 4, 5])->max();

    // 5

<a name="method-median"></a>
#### `median()`

Метод `median` возвращает [median value](https://en.wikipedia.org/wiki/Median) данного ключа:

    $median = collect([
        ['foo' => 10],
        ['foo' => 10],
        ['foo' => 20],
        ['foo' => 40]
    ])->median('foo');

    // 15

    $median = collect([1, 1, 2, 4])->median();

    // 1.5

<a name="method-merge"></a>
#### `merge()`

Метод `merge` объединяет данный массив или коллекцию с исходной коллекцией. Если строковый ключ в данных элементах соответствует строковому ключу в исходной коллекции, значение данного элемента перезапишет значение в исходной коллекции:

    $collection = collect(['product_id' => 1, 'price' => 100]);

    $merged = $collection->merge(['price' => 200, 'discount' => false]);

    $merged->all();

    // ['product_id' => 1, 'price' => 200, 'discount' => false]

Если ключи данного элемента являются числовыми, значения будут добавлены в конец коллекции:

    $collection = collect(['Desk', 'Chair']);

    $merged = $collection->merge(['Bookcase', 'Door']);

    $merged->all();

    // ['Desk', 'Chair', 'Bookcase', 'Door']

<a name="method-mergerecursive"></a>
#### `mergeRecursive()`

Метод `mergeRecursive` рекурсивно объединяет данный массив или коллекцию с исходной коллекцией. Если строковый ключ в заданных элементах совпадает со строковым ключом в исходной коллекции, то значения этих ключей объединяются в массив, и это делается рекурсивно:

    $collection = collect(['product_id' => 1, 'price' => 100]);

    $merged = $collection->mergeRecursive([
        'product_id' => 2,
        'price' => 200,
        'discount' => false
    ]);

    $merged->all();

    // ['product_id' => [1, 2], 'price' => [100, 200], 'discount' => false]

<a name="method-min"></a>
#### `min()`

Метод `min` возвращает минимальное значение данного ключа:

    $min = collect([['foo' => 10], ['foo' => 20]])->min('foo');

    // 10

    $min = collect([1, 2, 3, 4, 5])->min();

    // 1

<a name="method-mode"></a>
#### `mode()`

Метод `mode` возвращает  [mode value](https://en.wikipedia.org/wiki/Mode_(statistics)) данного ключа::

    $mode = collect([
        ['foo' => 10],
        ['foo' => 10],
        ['foo' => 20],
        ['foo' => 40]
    ])->mode('foo');

    // [10]

    $mode = collect([1, 1, 2, 4])->mode();

    // [1]

    $mode = collect([1, 1, 2, 2])->mode();

    // [1, 2]

<a name="method-nth"></a>
#### `nth()` 

Метод `nth` создает новую коллекцию, состоящую из каждого n-го элемента:

    $collection = collect(['a', 'b', 'c', 'd', 'e', 'f']);

    $collection->nth(4);

    // ['a', 'e']

При желании вы можете передать начальное смещение в качестве второго аргумента:

    $collection->nth(4, 1);

    // ['b', 'f']

<a name="method-only"></a>
#### `only()`

Метод `only` возвращает элементы коллекции с указанными ключами:

    $collection = collect([
        'product_id' => 1,
        'name' => 'Desk',
        'price' => 100,
        'discount' => false
    ]);

    $filtered = $collection->only(['product_id', 'name']);

    $filtered->all();

    // ['product_id' => 1, 'name' => 'Desk']

Для обратного `only`, см. [except](#method-except) метода.

<a name="method-pad"></a>
#### `pad()`

Метод `pad` будет заполнять массив заданным значением, пока массив не достигнет указанного размера. Этот метод ведет себя как PHP-функция [array_pad](https://secure.php.net/manual/en/function.array-pad.php).

Чтобы отступить влево, вы должны указать отрицательный размер. Заполнение не произойдет, если абсолютное значение заданного размера меньше или равно длине массива:

    $collection = collect(['A', 'B', 'C']);

    $filtered = $collection->pad(5, 0);

    $filtered->all();

    // ['A', 'B', 'C', 0, 0]

    $filtered = $collection->pad(-5, 0);

    $filtered->all();

    // [0, 0, 'A', 'B', 'C']

<a name="method-partition"></a>
#### `partition()`

Этот `partition` метод можно комбинировать с деструктуризацией массива PHP, чтобы отделить элементы, которые проходят заданный тест на истинность, от тех, которые не проходят:

    $collection = collect([1, 2, 3, 4, 5, 6]);

    [$underThree, $equalOrAboveThree] = $collection->partition(function (int $i) {
        return $i < 3;
    });

    $underThree->all();

    // [1, 2]

    $equalOrAboveThree->all();

    // [3, 4, 5, 6]

<a name="method-pipe"></a>
#### `pipe()`

Метод `pipe` передает коллекцию заданному замыканию и возвращает результат выполненного замыкания:

    $collection = collect([1, 2, 3]);

    $piped = $collection->pipe(function (Collection $collection) {
        return $collection->sum();
    });

    // 6

<a name="method-pipeinto"></a>
#### `pipeInto()`

Метод `pipeInto` создает новый экземпляр данного класса и передает коллекцию в конструктор:

    class ResourceCollection
    {
        /**
         * Create a new ResourceCollection instance.
         */
        public function __construct(
          public Collection $collection,
        ) {}
    }

    $collection = collect([1, 2, 3]);

    $resource = $collection->pipeInto(ResourceCollection::class);

    $resource->collection->all();

    // [1, 2, 3]

<a name="method-pipethrough"></a>
#### `pipeThrough()`

Метод `pipeThrough` передает коллекцию заданному массиву замыканий и возвращает результат выполненных замыканий:

    use Mnv\Core\Collections\Collection;

    $collection = collect([1, 2, 3]);

    $result = $collection->pipeThrough([
        function (Collection $collection) {
            return $collection->merge([4, 5]);
        },
        function (Collection $collection) {
            return $collection->sum();
        },
    ]);

    // 15

<a name="method-pluck"></a>
#### `pluck()`

Метод `pluck` извлекает все значения для данного ключа:

    $collection = collect([
        ['product_id' => 'prod-100', 'name' => 'Desk'],
        ['product_id' => 'prod-200', 'name' => 'Chair'],
    ]);

    $plucked = $collection->pluck('name');

    $plucked->all();

    // ['Desk', 'Chair']

Вы также можете указать, как вы хотите, чтобы результирующая коллекция была снабжена ключами:

    $plucked = $collection->pluck('name', 'product_id');

    $plucked->all();

    // ['prod-100' => 'Desk', 'prod-200' => 'Chair']

Метод `pluck` также поддерживает извлечение вложенных значений с использованием «точечной» нотации:

    $collection = collect([
        [
            'name' => 'Laracon',
            'speakers' => [
                'first_day' => ['Rosa', 'Judith'],
            ],
        ],
        [
            'name' => 'VueConf',
            'speakers' => [
                'first_day' => ['Abigail', 'Joey'],
            ],
        ],
    ]);

    $plucked = $collection->pluck('speakers.first_day');

    $plucked->all();

    // [['Rosa', 'Judith'], ['Abigail', 'Joey']]

Если дублирующиеся ключи существуют, последний совпадающий элемент будет вставлен в извлеченную коллекцию:

    $collection = collect([
        ['brand' => 'Tesla',  'color' => 'red'],
        ['brand' => 'Pagani', 'color' => 'white'],
        ['brand' => 'Tesla',  'color' => 'black'],
        ['brand' => 'Pagani', 'color' => 'orange'],
    ]);

    $plucked = $collection->pluck('color', 'brand');

    $plucked->all();

    // ['Tesla' => 'black', 'Pagani' => 'orange']

<a name="method-pop"></a>
#### `pop()`

Метод `pop` удаляет и возвращает последний элемент из коллекции:

    $collection = collect([1, 2, 3, 4, 5]);

    $collection->pop();

    // 5

    $collection->all();

    // [1, 2, 3, 4]

Вы можете передать методу целое число `pop`, чтобы удалить и вернуть несколько элементов из конца коллекции:

    $collection = collect([1, 2, 3, 4, 5]);

    $collection->pop(3);

    // collect([5, 4, 3])

    $collection->all();

    // [1, 2]

<a name="method-prepend"></a>
#### `prepend()`

Метод `prepend` добавляет элемент в начало коллекции:

    $collection = collect([1, 2, 3, 4, 5]);

    $collection->prepend(0);

    $collection->all();

    // [0, 1, 2, 3, 4, 5]

Вы также можете передать второй аргумент, чтобы указать ключ добавленного элемента:

    $collection = collect(['one' => 1, 'two' => 2]);

    $collection->prepend(0, 'zero');

    $collection->all();

    // ['zero' => 0, 'one' => 1, 'two' => 2]

<a name="method-pull"></a>
#### `pull()`

Метод `pull` удаляет и возвращает элемент из коллекции по его ключу:

    $collection = collect(['product_id' => 'prod-100', 'name' => 'Desk']);

    $collection->pull('name');

    // 'Desk'

    $collection->all();

    // ['product_id' => 'prod-100']

<a name="method-push"></a>
#### `push()`

Метод `push` добавляет элемент в конец коллекции:

    $collection = collect([1, 2, 3, 4]);

    $collection->push(5);

    $collection->all();

    // [1, 2, 3, 4, 5]

<a name="method-put"></a>
#### `put()`

Метод `put` устанавливает заданный ключ и значение в коллекции:

    $collection = collect(['product_id' => 1, 'name' => 'Desk']);

    $collection->put('price', 100);

    $collection->all();

    // ['product_id' => 1, 'name' => 'Desk', 'price' => 100]

<a name="method-random"></a>
#### `random()`

Метод `random` возвращает случайный элемент из коллекции:

    $collection = collect([1, 2, 3, 4, 5]);

    $collection->random();

    // 4 - (retrieved randomly)

Вы можете передать целое число, `random` чтобы указать, сколько элементов вы хотите получить случайным образом. Коллекция элементов всегда возвращается при явной передаче количества элементов, которые вы хотите получить:

    $random = $collection->random(3);

    $random->all();

    // [2, 4, 5] - (retrieved randomly)

Если в экземпляре коллекции меньше элементов, чем запрошено, `random` метод выдаст ошибку `InvalidArgumentException`.

Метод `random` также принимает замыкание, которое получит текущий экземпляр коллекции:

    use Mnv\Core\Collections\Collection;

    $random = $collection->random(fn (Collection $items) => min(10, count($items)));

    $random->all();

    // [1, 2, 3, 4, 5] - (retrieved randomly)

<a name="method-range"></a>
#### `range()`

Метод `range` возвращает коллекцию, содержащую целые числа из указанного диапазона:

    $collection = collect()->range(3, 6);

    $collection->all();

    // [3, 4, 5, 6]

<a name="method-reduce"></a>
#### `reduce()`

Метод `reduce` сводит коллекцию к одному значению, передавая результат каждой итерации в последующую итерацию:

    $collection = collect([1, 2, 3]);

    $total = $collection->reduce(function (int $carry, int $item) {
        return $carry + $item;
    });

    // 6

Значение для `$carry` первой итерации равно `null`; однако вы можете указать его начальное значение, передав второй аргумент `reduce`:

    $collection->reduce(function (int $carry, int $item) {
        return $carry + $item;
    }, 4);

    // 10

Метод `reduce` также передает ключи массива в ассоциативных коллекциях данному обратному вызову:

    $collection = collect([
        'usd' => 1400,
        'gbp' => 1200,
        'eur' => 1000,
    ]);

    $ratio = [
        'usd' => 1,
        'gbp' => 1.37,
        'eur' => 1.22,
    ];

    $collection->reduce(function (int $carry, int $value, int $key) use ($ratio) {
        return $carry + ($value * $ratio[$key]);
    });

    // 4264
    
<a name="method-reduce-spread"></a>
#### `reduceSpread()`

Метод `reduceSpread` сводит коллекцию к массиву значений, передавая результаты каждой итерации в последующую итерацию. Этот метод аналогичен методу `reduce`; однако он может принимать несколько начальных значений:

    [$creditsRemaining, $batch] = Image::where('status', 'unprocessed')
        ->get()
        ->reduceSpread(function (int $creditsRemaining, Collection $batch, Image $image) {
            if ($creditsRemaining >= $image->creditsRequired()) {
                $batch->push($image);

                $creditsRemaining -= $image->creditsRequired();
            }

            return [$creditsRemaining, $batch];
        }, $creditsAvailable, collect());

<a name="method-reject"></a>
#### `reject()`

Метод `reject` фильтрует коллекцию, используя данное замыкание. 
Закрытие должно вернуться `true`, если элемент должен быть удален из результирующей коллекции:

    $collection = collect([1, 2, 3, 4]);

    $filtered = $collection->reject(function (int $value, int $key) {
        return $value > 2;
    });

    $filtered->all();

    // [1, 2]

Об обратном методе `reject` см. [`filter`](#method-filter) метод.

<a name="method-replace"></a>
#### `replace()`

Метод `replace` ведет себя аналогично `merge`; однако в дополнение к перезаписи совпадающих элементов со строковыми ключами метод `replace` также перезапишет элементы в коллекции с совпадающими числовыми ключами:

    $collection = collect(['Taylor', 'Abigail', 'James']);

    $replaced = $collection->replace([1 => 'Victoria', 3 => 'Finn']);

    $replaced->all();

    // ['Taylor', 'Victoria', 'James', 'Finn']

<a name="method-replacerecursive"></a>
#### `replaceRecursive()`

Этот метод работает как `replace`, но он будет повторяться в массивах и применять тот же процесс замены к внутренним значениям:

    $collection = collect([
        'Taylor',
        'Abigail',
        [
            'James',
            'Victoria',
            'Finn'
        ]
    ]);

    $replaced = $collection->replaceRecursive([
        'Charlie',
        2 => [1 => 'King']
    ]);

    $replaced->all();

    // ['Charlie', 'Abigail', ['James', 'King', 'Finn']]

<a name="method-reverse"></a>
#### `reverse()`

Метод `reverse` меняет порядок элементов коллекции, сохраняя исходные ключи:

    $collection = collect(['a', 'b', 'c', 'd', 'e']);

    $reversed = $collection->reverse();

    $reversed->all();

    /*
        [
            4 => 'e',
            3 => 'd',
            2 => 'c',
            1 => 'b',
            0 => 'a',
        ]
    */

<a name="method-search"></a>
#### `search()`

Метод `search` ищет в коллекции заданное значение и возвращает его ключ, если он найден. Если элемент не найден, `false` возвращается:

    $collection = collect([2, 4, 6, 8]);

    $collection->search(4);

    // 1

Поиск выполняется с использованием «свободного» сравнения, то есть строка с целочисленным значением будет считаться равной целому числу того же значения. Чтобы использовать «строгое» сравнение, передайте `true` в качестве второго аргумента методу:

    collect([2, 4, 6, 8])->search('4', $strict = true);

    // false

В качестве альтернативы вы можете предоставить собственное замыкание для поиска первого элемента, прошедшего заданный тест на истинность:

    collect([2, 4, 6, 8])->search(function (int $item, int $key) {
        return $item > 5;
    });

    // 2

<a name="method-shift"></a>
#### `shift()`

Метод `shift` удаляет и возвращает первый элемент из коллекции:

    $collection = collect([1, 2, 3, 4, 5]);

    $collection->shift();

    // 1

    $collection->all();

    // [2, 3, 4, 5]

Вы можете передать методу целое число `shift`, чтобы удалить и вернуть несколько элементов из начала коллекции:

    $collection = collect([1, 2, 3, 4, 5]);

    $collection->shift(3);

    // collect([1, 2, 3])

    $collection->all();

    // [4, 5]

<a name="method-shuffle"></a>
#### `shuffle()`

Метод `shuffle` случайным образом перемешивает элементы в коллекции:

    $collection = collect([1, 2, 3, 4, 5]);

    $shuffled = $collection->shuffle();

    $shuffled->all();

    // [3, 2, 5, 1, 4] - (generated randomly)

<a name="method-skip"></a>
#### `skip()`

Метод `skip` возвращает новую коллекцию с заданным количеством элементов, удаленных из начала коллекции:

    $collection = collect([1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);

    $collection = $collection->skip(4);

    $collection->all();

    // [5, 6, 7, 8, 9, 10]

<a name="method-skipuntil"></a>
#### `skipUntil()` 

Метод `skipUntil` пропускает элементы из коллекции до тех пор, пока данный обратный вызов не вернется, `true` а затем возвращает оставшиеся элементы в коллекции как новый экземпляр коллекции:

    $collection = collect([1, 2, 3, 4]);

    $subset = $collection->skipUntil(function (int $item) {
        return $item >= 3;
    });

    $subset->all();

    // [3, 4]

Вы также можете передать методу простое значение, `skipUntil` чтобы пропустить все элементы, пока не будет найдено заданное значение:

    $collection = collect([1, 2, 3, 4]);

    $subset = $collection->skipUntil(3);

    $subset->all();

    // [3, 4]

> **Предупреждение**  
> Если заданное значение не найдено или обратный вызов никогда не возвращает значение `true`, `skipUntil` метод вернет пустую коллекцию.

<a name="method-skipwhile"></a>
#### `skipWhile()`

Метод `skipWhile` пропускает элементы из коллекции, пока возвращается данный обратный вызов, `true` а затем возвращает оставшиеся элементы в коллекции как новую коллекцию:

    $collection = collect([1, 2, 3, 4]);

    $subset = $collection->skipWhile(function (int $item) {
        return $item <= 3;
    });

    $subset->all();

    // [4]

> **Предупреждение**  
> Если обратный вызов никогда не возвращает значение `false`, `skipWhile` метод вернет пустую коллекцию.

<a name="method-slice"></a>
#### `slice()`

Метод `slice` возвращает часть коллекции, начиная с заданного индекса:

    $collection = collect([1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);

    $slice = $collection->slice(4);

    $slice->all();

    // [5, 6, 7, 8, 9, 10]

Если вы хотите ограничить размер возвращаемого фрагмента, передайте требуемый размер в качестве второго аргумента метода:

    $slice = $collection->slice(4, 2);

    $slice->all();

    // [5, 6]

Возвращенный фрагмент будет сохранять ключи по умолчанию. Если вы не хотите сохранять исходные ключи, вы можете использовать [`values`](#method-values) метод их переиндексации.

<a name="method-sliding"></a>
#### `sliding()` 

Метод `sliding` возвращает новую коллекцию фрагментов, представляющих «скользящее окно» элементов коллекции:

    $collection = collect([1, 2, 3, 4, 5]);

    $chunks = $collection->sliding(2);

    $chunks->toArray();

    // [[1, 2], [2, 3], [3, 4], [4, 5]]

Это особенно полезно в сочетании с [`eachSpread`](#method-eachspread) методом:

    $transactions->sliding(2)->eachSpread(function (Collection $previous, Collection $current) {
        $current->total = $previous->total + $current->amount;
    });

При желании вы можете передать второе значение «шага», которое определяет расстояние между первым элементом каждого фрагмента:

    $collection = collect([1, 2, 3, 4, 5]);

    $chunks = $collection->sliding(3, step: 2);

    $chunks->toArray();

    // [[1, 2, 3], [3, 4, 5]]

<a name="method-sole"></a>
#### `sole()`

Метод `sole` возвращает первый элемент в коллекции, который проходит заданный тест на истинность, но только если тест на истинность соответствует ровно одному элементу:

    collect([1, 2, 3, 4])->sole(function (int $value, int $key) {
        return $value === 2;
    });

    // 2

Вы также можете передать пару ключ/значение в `sole` метод, который вернет первый элемент в коллекции, который соответствует заданной паре, но только если совпадает ровно один элемент:

    $collection = collect([
        ['product' => 'Desk', 'price' => 200],
        ['product' => 'Chair', 'price' => 100],
    ]);

    $collection->sole('product', 'Chair');

    // ['product' => 'Chair', 'price' => 100]

В качестве альтернативы вы также можете вызвать `sole` метод без аргументов, чтобы получить первый элемент в коллекции, если есть только один элемент:

    $collection = collect([
        ['product' => 'Desk', 'price' => 200],
    ]);

    $collection->sole();

    // ['product' => 'Desk', 'price' => 200]

Если в коллекции нет элементов, которые должны быть возвращены методом `sole`, \Mnv\Core\Collections\ItemNotFoundException будет выдано исключение. 
Если должно быть возвращено более одного элемента, \Mnv\Core\Collections\MultipleItemsFoundException будет выброшено.

<a name="method-some"></a>
#### `some()`

Псевдоним метода [`contains`](#method-contains).

<a name="method-sort"></a>
#### `sort()` 

Метод `sort` сортирует коллекцию. Отсортированная коллекция сохраняет исходные ключи массива, поэтому в следующем примере мы будем использовать [`values`](#method-values) метод для сброса ключей к последовательно пронумерованным индексам:

    $collection = collect([5, 3, 1, 2, 4]);

    $sorted = $collection->sort();

    $sorted->values()->all();

    // [1, 2, 3, 4, 5]

Если ваши потребности в сортировке более сложны, вы можете передать обратный вызов `sort` со своим собственным алгоритмом. Обратитесь к документации по PHP [`uasort`](https://secure.php.net/manual/en/function.uasort.php#refsect1-function.uasort-parameters), которая используется внутри вызовов методов коллекции sort.

> **Примечание**  
> Если вам нужно отсортировать коллекцию вложенных массивов или объектов, см. методы [`sortBy`](#method-sortby) и [`sortByDesc`](#method-sortbydesc).

<a name="method-sortby"></a>
#### `sortBy()`

Метод `sortBy` сортирует коллекцию по заданному ключу. Отсортированная коллекция сохраняет исходные ключи массива, поэтому в следующем примере мы будем использовать [`values`](#method-values) метод для сброса ключей к последовательно пронумерованным индексам:

    $collection = collect([
        ['name' => 'Desk', 'price' => 200],
        ['name' => 'Chair', 'price' => 100],
        ['name' => 'Bookcase', 'price' => 150],
    ]);

    $sorted = $collection->sortBy('price');

    $sorted->values()->all();

    /*
        [
            ['name' => 'Chair', 'price' => 100],
            ['name' => 'Bookcase', 'price' => 150],
            ['name' => 'Desk', 'price' => 200],
        ]
    */

Метод `sortBy` принимает [sort flags](https://www.php.net/manual/en/function.sort.php) в качестве второго аргумента:

    $collection = collect([
        ['title' => 'Item 1'],
        ['title' => 'Item 12'],
        ['title' => 'Item 3'],
    ]);

    $sorted = $collection->sortBy('title', SORT_NATURAL);

    $sorted->values()->all();

    /*
        [
            ['title' => 'Item 1'],
            ['title' => 'Item 3'],
            ['title' => 'Item 12'],
        ]
    */

В качестве альтернативы вы можете передать собственное замыкание, чтобы определить, как сортировать значения коллекции:

    $collection = collect([
        ['name' => 'Desk', 'colors' => ['Black', 'Mahogany']],
        ['name' => 'Chair', 'colors' => ['Black']],
        ['name' => 'Bookcase', 'colors' => ['Red', 'Beige', 'Brown']],
    ]);

    $sorted = $collection->sortBy(function (array $product, int $key) {
        return count($product['colors']);
    });

    $sorted->values()->all();

    /*
        [
            ['name' => 'Chair', 'colors' => ['Black']],
            ['name' => 'Desk', 'colors' => ['Black', 'Mahogany']],
            ['name' => 'Bookcase', 'colors' => ['Red', 'Beige', 'Brown']],
        ]
    */

Если вы хотите отсортировать свою коллекцию по нескольким атрибутам, вы можете передать методу массив операций сортировки `sortBy`. Каждая операция сортировки должна быть массивом, состоящим из атрибута, по которому вы хотите отсортировать, и направления желаемой сортировки:

    $collection = collect([
        ['name' => 'Taylor Otwell', 'age' => 34],
        ['name' => 'Abigail Otwell', 'age' => 30],
        ['name' => 'Taylor Otwell', 'age' => 36],
        ['name' => 'Abigail Otwell', 'age' => 32],
    ]);

    $sorted = $collection->sortBy([
        ['name', 'asc'],
        ['age', 'desc'],
    ]);

    $sorted->values()->all();

    /*
        [
            ['name' => 'Abigail Otwell', 'age' => 32],
            ['name' => 'Abigail Otwell', 'age' => 30],
            ['name' => 'Taylor Otwell', 'age' => 36],
            ['name' => 'Taylor Otwell', 'age' => 34],
        ]
    */

При сортировке коллекции по нескольким атрибутам вы также можете предоставить замыкания, определяющие каждую операцию сортировки:

    $collection = collect([
        ['name' => 'Taylor Otwell', 'age' => 34],
        ['name' => 'Abigail Otwell', 'age' => 30],
        ['name' => 'Taylor Otwell', 'age' => 36],
        ['name' => 'Abigail Otwell', 'age' => 32],
    ]);

    $sorted = $collection->sortBy([
        fn (array $a, array $b) => $a['name'] <=> $b['name'],
        fn (array $a, array $b) => $b['age'] <=> $a['age'],
    ]);

    $sorted->values()->all();

    /*
        [
            ['name' => 'Abigail Otwell', 'age' => 32],
            ['name' => 'Abigail Otwell', 'age' => 30],
            ['name' => 'Taylor Otwell', 'age' => 36],
            ['name' => 'Taylor Otwell', 'age' => 34],
        ]
    */

<a name="method-sortbydesc"></a>
#### `sortByDesc()`

Этот метод имеет ту же сигнатуру, что и [`sortBy`](#method-sortby) метод, но сортирует коллекцию в обратном порядке.

<a name="method-sortdesc"></a>
#### `sortDesc()`

Этот метод будет сортировать коллекцию в порядке, обратном методу [`sort`](#method-sort):

    $collection = collect([5, 3, 1, 2, 4]);

    $sorted = $collection->sortDesc();

    $sorted->values()->all();

    // [5, 4, 3, 2, 1]

В отличие от sort, вы не можете передать замыкание в sortDesc. Вместо этого вы должны использовать [`sort`](#method-sort) метод и инвертировать сравнение.

<a name="method-sortkeys"></a>
#### `sortKeys()`

Метод `sortKeys` сортирует коллекцию по ключам базового ассоциативного массива:

    $collection = collect([
        'id' => 22345,
        'first' => 'John',
        'last' => 'Doe',
    ]);

    $sorted = $collection->sortKeys();

    $sorted->all();

    /*
        [
            'first' => 'John',
            'id' => 22345,
            'last' => 'Doe',
        ]
    */

<a name="method-sortkeysdesc"></a>
#### `sortKeysDesc()`

Этот метод имеет ту же сигнатуру, что и [`sortKeys`](#method-sortkeys) метод, но сортирует коллекцию в обратном порядке.

<a name="method-sortkeysusing"></a>
#### `sortKeysUsing()`

Метод `sortKeysUsing` сортирует коллекцию по ключам базового ассоциативного массива с помощью обратного вызова:

    $collection = collect([
        'ID' => 22345,
        'first' => 'John',
        'last' => 'Doe',
    ]);

    $sorted = $collection->sortKeysUsing('strnatcasecmp');

    $sorted->all();

    /*
        [
            'first' => 'John',
            'ID' => 22345,
            'last' => 'Doe',
        ]
    */

Обратный вызов должен быть функцией сравнения, которая возвращает целое число, меньшее, равное или большее нуля. Для получения дополнительной информации обратитесь к документации по PHP [`uksort`](https://www.php.net/manual/en/function.uksort.php#refsect1-function.uksort-parameters),  которая представляет собой функцию PHP, `sortKeysUsing` используемую внутри метода.

<a name="method-splice"></a>
#### `splice()`

Метод `splice` удаляет и возвращает фрагмент элементов, начиная с указанного индекса:

    $collection = collect([1, 2, 3, 4, 5]);

    $chunk = $collection->splice(2);

    $chunk->all();

    // [3, 4, 5]

    $collection->all();

    // [1, 2]

Вы можете передать второй аргумент, чтобы ограничить размер результирующей коллекции:

    $collection = collect([1, 2, 3, 4, 5]);

    $chunk = $collection->splice(2, 1);

    $chunk->all();

    // [3]

    $collection->all();

    // [1, 2, 4, 5]

Кроме того, вы можете передать третий аргумент, содержащий новые элементы, чтобы заменить элементы, удаленные из коллекции:

    $collection = collect([1, 2, 3, 4, 5]);

    $chunk = $collection->splice(2, 1, [10, 11]);

    $chunk->all();

    // [3]

    $collection->all();

    // [1, 2, 10, 11, 4, 5]

<a name="method-split"></a>
#### `split()`

Метод `split` разбивает коллекцию на заданное количество групп:

    $collection = collect([1, 2, 3, 4, 5]);

    $groups = $collection->split(3);

    $groups->all();

    // [[1, 2], [3, 4], [5]]

<a name="method-splitin"></a>
#### `splitIn()`

Метод `splitIn` разбивает коллекцию на заданное количество групп, полностью заполняя неконечные группы, прежде чем выделить остаток в последнюю группу:

    $collection = collect([1, 2, 3, 4, 5, 6, 7, 8, 9, 10]);

    $groups = $collection->splitIn(3);

    $groups->all();

    // [[1, 2, 3, 4], [5, 6, 7, 8], [9, 10]]

<a name="method-sum"></a>
#### `sum()`

Метод `sum` возвращает сумму всех элементов коллекции:

    collect([1, 2, 3, 4, 5])->sum();

    // 15

Если коллекция содержит вложенные массивы или объекты, вы должны передать ключ, который будет использоваться для определения суммируемых значений:

    $collection = collect([
        ['name' => 'JavaScript: The Good Parts', 'pages' => 176],
        ['name' => 'JavaScript: The Definitive Guide', 'pages' => 1096],
    ]);

    $collection->sum('pages');

    // 1272

Кроме того, вы можете передать собственное замыкание, чтобы определить, какие значения коллекции суммировать:

    $collection = collect([
        ['name' => 'Chair', 'colors' => ['Black']],
        ['name' => 'Desk', 'colors' => ['Black', 'Mahogany']],
        ['name' => 'Bookcase', 'colors' => ['Red', 'Beige', 'Brown']],
    ]);

    $collection->sum(function (array $product) {
        return count($product['colors']);
    });

    // 6

<a name="method-take"></a>
#### `take()`

Метод `take` возвращает новую коллекцию с указанным количеством элементов:

    $collection = collect([0, 1, 2, 3, 4, 5]);

    $chunk = $collection->take(3);

    $chunk->all();

    // [0, 1, 2]

Вы также можете передать отрицательное целое число, чтобы взять указанное количество элементов с конца коллекции:

    $collection = collect([0, 1, 2, 3, 4, 5]);

    $chunk = $collection->take(-2);

    $chunk->all();

    // [4, 5]

<a name="method-takeuntil"></a>
#### `takeUntil()`

Метод `takeUntil` возвращает элементы в коллекции до тех пор, пока данный обратный вызов не вернет `true`:

    $collection = collect([1, 2, 3, 4]);

    $subset = $collection->takeUntil(function (int $item) {
        return $item >= 3;
    });

    $subset->all();

    // [1, 2]

Вы также можете передать методу простое значение, `takeUntil` чтобы получить элементы, пока не будет найдено заданное значение:

    $collection = collect([1, 2, 3, 4]);

    $subset = $collection->takeUntil(3);

    $subset->all();

    // [1, 2]

> **Предупреждение**  
> Если заданное значение не найдено или обратный вызов никогда не возвращает значение `true`, `takeUntil` метод вернет все элементы коллекции.

<a name="method-takewhile"></a>
#### `takeWhile()`

Метод `takeWhile` возвращает элементы в коллекции до тех пор, пока данный обратный вызов не вернет `false`:

    $collection = collect([1, 2, 3, 4]);

    $subset = $collection->takeWhile(function (int $item) {
        return $item < 3;
    });

    $subset->all();

    // [1, 2]

> **Предупреждение**  
> Если обратный вызов никогда не возвращает значение `false`, `takeWhile` метод вернет все элементы коллекции.

<a name="method-tap"></a>
#### `tap()`

Метод `tap` передает коллекцию заданному обратному вызову, позволяя вам «подключиться» к коллекции в определенной точке и сделать что-то с элементами, не затрагивая саму коллекцию. Затем коллекция возвращается методом `tap`:

    collect([2, 4, 3, 1, 5])
        ->sort()
        ->tap(function (Collection $collection) {
            Log::debug('Values after sorting', $collection->values()->all());
        })
        ->shift();

    // 1

<a name="method-times"></a>
#### `times()`

Статический `times` метод создает новую коллекцию, вызывая данное замыкание указанное количество раз:

    $collection = Collection::times(10, function (int $number) {
        return $number * 9;
    });

    $collection->all();

    // [9, 18, 27, 36, 45, 54, 63, 72, 81, 90]

<a name="method-toarray"></a>
#### `toArray()`

Метод `toArray` преобразует коллекцию в обычный PHP `array`. Если значениями коллекции являются модели [Eloquent](/docs/eloquent), модели также будут преобразованы в массивы:

    $collection = collect(['name' => 'Desk', 'price' => 200]);

    $collection->toArray();

    /*
        [
            ['name' => 'Desk', 'price' => 200],
        ]
    */

> **Предупреждение**  
> `toArray` также преобразует все вложенные объекты коллекции, которые являются экземплярами, `Arrayable` в массив. Если вы хотите получить необработанный массив, лежащий в основе коллекции, используйте [`all`](#method-all) вместо этого метод.

<a name="method-tojson"></a>
#### `toJson()`

Метод `toJson` преобразует коллекцию в сериализованную строку `JSON`:

    $collection = collect(['name' => 'Desk', 'price' => 200]);

    $collection->toJson();

    // '{"name":"Desk", "price":200}'

<a name="method-transform"></a>
#### `transform()`

Метод `transform` выполняет итерацию по коллекции и вызывает заданный обратный вызов для каждого элемента коллекции. Элементы в коллекции будут заменены значениями, возвращаемыми обратным вызовом:

    $collection = collect([1, 2, 3, 4, 5]);

    $collection->transform(function (int $item, int $key) {
        return $item * 2;
    });

    $collection->all();

    // [2, 4, 6, 8, 10]

> **Предупреждение**  
> В отличие от большинства других методов сбора, `transform` модифицирует саму коллекцию. Если вы хотите вместо этого создать новую коллекцию, используйте [`map`](#method-map) метод.

<a name="method-undot"></a>
#### `undot()`

Метод `undot` расширяет одномерную коллекцию, использующую «точечную» нотацию, в многомерную коллекцию:

    $person = collect([
        'name.first_name' => 'Marie',
        'name.last_name' => 'Valentine',
        'address.line_1' => '2992 Eagle Drive',
        'address.line_2' => '',
        'address.suburb' => 'Detroit',
        'address.state' => 'MI',
        'address.postcode' => '48219'
    ]);

    $person = $person->undot();

    $person->toArray();

    /*
        [
            "name" => [
                "first_name" => "Marie",
                "last_name" => "Valentine",
            ],
            "address" => [
                "line_1" => "2992 Eagle Drive",
                "line_2" => "",
                "suburb" => "Detroit",
                "state" => "MI",
                "postcode" => "48219",
            ],
        ]
    */

<a name="method-union"></a>
#### `union()`

Метод `union` добавляет заданный массив в коллекцию. Если данный массив содержит ключи, которые уже есть в исходной коллекции, предпочтительными будут значения исходной коллекции:

    $collection = collect([1 => ['a'], 2 => ['b']]);

    $union = $collection->union([3 => ['c'], 1 => ['d']]);

    $union->all();

    // [1 => ['a'], 2 => ['b'], 3 => ['c']]

<a name="method-unique"></a>
#### `unique()`

Метод `unique` возвращает все уникальные элементы коллекции. Возвращаемая коллекция сохраняет исходные ключи массива, поэтому в следующем примере мы будем использовать [`values`](#method-values) метод для сброса ключей к последовательно пронумерованным индексам:

    $collection = collect([1, 1, 2, 2, 3, 4, 2]);

    $unique = $collection->unique();

    $unique->values()->all();

    // [1, 2, 3, 4]

При работе с вложенными массивами или объектами вы можете указать ключ, используемый для определения уникальности:

    $collection = collect([
        ['name' => 'iPhone 6', 'brand' => 'Apple', 'type' => 'phone'],
        ['name' => 'iPhone 5', 'brand' => 'Apple', 'type' => 'phone'],
        ['name' => 'Apple Watch', 'brand' => 'Apple', 'type' => 'watch'],
        ['name' => 'Galaxy S6', 'brand' => 'Samsung', 'type' => 'phone'],
        ['name' => 'Galaxy Gear', 'brand' => 'Samsung', 'type' => 'watch'],
    ]);

    $unique = $collection->unique('brand');

    $unique->values()->all();

    /*
        [
            ['name' => 'iPhone 6', 'brand' => 'Apple', 'type' => 'phone'],
            ['name' => 'Galaxy S6', 'brand' => 'Samsung', 'type' => 'phone'],
        ]
    */

Наконец, вы также можете передать собственное замыкание в `unique` метод, чтобы указать, какое значение должно определять уникальность элемента:

    $unique = $collection->unique(function (array $item) {
        return $item['brand'].$item['type'];
    });

    $unique->values()->all();

    /*
        [
            ['name' => 'iPhone 6', 'brand' => 'Apple', 'type' => 'phone'],
            ['name' => 'Apple Watch', 'brand' => 'Apple', 'type' => 'watch'],
            ['name' => 'Galaxy S6', 'brand' => 'Samsung', 'type' => 'phone'],
            ['name' => 'Galaxy Gear', 'brand' => 'Samsung', 'type' => 'watch'],
        ]
    */

Метод `unique` использует «свободные» сравнения при проверке значений элементов, то есть строка с целочисленным значением будет считаться равной целому числу с тем же значением. Используйте [`uniqueStrict`](#method-uniquestrict) метод для фильтрации с использованием «строгих» сравнений.

> **Примечание**  
> Поведение этого метода изменяется при использовании [Eloquent Collections](/docs/{{version}}/eloquent-collections#method-unique).

<a name="method-uniquestrict"></a>
#### `uniqueStrict()`

Этот метод имеет ту же сигнатуру, что и [`unique`](#method-unique) метод; однако все значения сравниваются с использованием «строгих» сравнений.

<a name="method-unless"></a>
#### `unless()`

Метод `unless` выполнит данный обратный вызов, если только первый аргумент, переданный методу, не будет равен `true`:

    $collection = collect([1, 2, 3]);

    $collection->unless(true, function (Collection $collection) {
        return $collection->push(4);
    });

    $collection->unless(false, function (Collection $collection) {
        return $collection->push(5);
    });

    $collection->all();

    // [1, 2, 3, 5]

В метод может быть передан второй обратный вызов `unless`. Второй обратный вызов будет выполнен, когда первый аргумент, переданный методу, будет `unless` равен `true`:

    $collection = collect([1, 2, 3]);

    $collection->unless(true, function (Collection $collection) {
        return $collection->push(4);
    }, function (Collection $collection) {
        return $collection->push(5);
    });

    $collection->all();

    // [1, 2, 3, 5]

Для обратного `unless` см. [`when`](#method-when) метод.

<a name="method-unlessempty"></a>
#### `unlessEmpty()`

Псевдоним метода [`whenNotEmpty`](#method-whennotempty).

<a name="method-unlessnotempty"></a>
#### `unlessNotEmpty()`

Псевдоним метода [`whenEmpty`](#method-whenempty).

<a name="method-unwrap"></a>
#### `unwrap()`

Статический `unwrap` метод возвращает базовые элементы коллекции из заданного значения, когда это применимо:

    Collection::unwrap(collect('John Doe'));

    // ['John Doe']

    Collection::unwrap(['John Doe']);

    // ['John Doe']

    Collection::unwrap('John Doe');

    // 'John Doe'

<a name="method-value"></a>
#### `value()`

Метод `value` извлекает заданное значение из первого элемента коллекции:

    $collection = collect([
        ['product' => 'Desk', 'price' => 200],
        ['product' => 'Speaker', 'price' => 400],
    ]);

    $value = $collection->value('price');

    // 200

<a name="method-values"></a>
#### `values()`

Метод `values` возвращает новую коллекцию с ключами, сброшенными на последовательные целые числа:

    $collection = collect([
        10 => ['product' => 'Desk', 'price' => 200],
        11 => ['product' => 'Desk', 'price' => 200],
    ]);

    $values = $collection->values();

    $values->all();

    /*
        [
            0 => ['product' => 'Desk', 'price' => 200],
            1 => ['product' => 'Desk', 'price' => 200],
        ]
    */

<a name="method-when"></a>
#### `when()`

Метод `when` выполнит данный обратный вызов, когда первый аргумент, переданный методу, будет равен `true`. Экземпляр коллекции и первый аргумент, переданный методу, `when` будут переданы замыканию:

    $collection = collect([1, 2, 3]);

    $collection->when(true, function (Collection $collection, int $value) {
        return $collection->push(4);
    });

    $collection->when(false, function (Collection $collection, int $value) {
        return $collection->push(5);
    });

    $collection->all();

    // [1, 2, 3, 4]

В метод может быть передан второй обратный вызов `when`. Второй обратный вызов будет выполнен, когда первый аргумент, переданный методу, будет `when` равен `false`:

    $collection = collect([1, 2, 3]);

    $collection->when(false, function (Collection $collection, int $value) {
        return $collection->push(4);
    }, function (Collection $collection) {
        return $collection->push(5);
    });

    $collection->all();

    // [1, 2, 3, 5]

Для обратного `when` см.  [`unless`](#method-unless) метод.

<a name="method-whenempty"></a>
#### `whenEmpty()`

Метод `whenEmpty` выполнит данный обратный вызов, когда коллекция пуста:

    $collection = collect(['Michael', 'Tom']);

    $collection->whenEmpty(function (Collection $collection) {
        return $collection->push('Adam');
    });

    $collection->all();

    // ['Michael', 'Tom']


    $collection = collect();

    $collection->whenEmpty(function (Collection $collection) {
        return $collection->push('Adam');
    });

    $collection->all();

    // ['Adam']

Второе замыкание может быть передано в `whenEmpty` метод, который будет выполняться, когда коллекция не пуста:

    $collection = collect(['Michael', 'Tom']);

    $collection->whenEmpty(function (Collection $collection) {
        return $collection->push('Adam');
    }, function (Collection $collection) {
        return $collection->push('Taylor');
    });

    $collection->all();

    // ['Michael', 'Tom', 'Taylor']

Для обратного `whenEmpty` см. [`whenNotEmpty`](#method-whennotempty).

<a name="method-whennotempty"></a>
#### `whenNotEmpty()`

Метод `whenNotEmpty` выполнит данный обратный вызов, когда коллекция не пуста:

    $collection = collect(['michael', 'tom']);

    $collection->whenNotEmpty(function (Collection $collection) {
        return $collection->push('adam');
    });

    $collection->all();

    // ['michael', 'tom', 'adam']


    $collection = collect();

    $collection->whenNotEmpty(function (Collection $collection) {
        return $collection->push('adam');
    });

    $collection->all();

    // []

Второе замыкание может быть передано в `whenNotEmpty` метод, который будет выполняться, когда коллекция пуста:

    $collection = collect();

    $collection->whenNotEmpty(function (Collection $collection) {
        return $collection->push('adam');
    }, function (Collection $collection) {
        return $collection->push('taylor');
    });

    $collection->all();

    // ['taylor']

Для обратного `whenNotEmpty` см.  [`whenEmpty`](#method-whenempty) метод.

<a name="method-where"></a>
#### `where()`

Метод `where` фильтрует коллекцию по заданной паре ключ/значение:

    $collection = collect([
        ['product' => 'Desk', 'price' => 200],
        ['product' => 'Chair', 'price' => 100],
        ['product' => 'Bookcase', 'price' => 150],
        ['product' => 'Door', 'price' => 100],
    ]);

    $filtered = $collection->where('price', 100);

    $filtered->all();

    /*
        [
            ['product' => 'Chair', 'price' => 100],
            ['product' => 'Door', 'price' => 100],
        ]
    */

Метод `where` использует «свободные» сравнения при проверке значений элементов, то есть строка с целочисленным значением будет считаться равной целому числу с тем же значением. Используйте [`whereStrict`](#method-wherestrict) метод для фильтрации с использованием «строгих» сравнений.

При желании вы можете передать оператор сравнения в качестве второго параметра. Поддерживаются следующие операторы: '===', '!==', '!=', '==', '=', '<>', '>', '<', '>=', и '<=':

    $collection = collect([
        ['name' => 'Jim', 'deleted_at' => '2019-01-01 00:00:00'],
        ['name' => 'Sally', 'deleted_at' => '2019-01-02 00:00:00'],
        ['name' => 'Sue', 'deleted_at' => null],
    ]);

    $filtered = $collection->where('deleted_at', '!=', null);

    $filtered->all();

    /*
        [
            ['name' => 'Jim', 'deleted_at' => '2019-01-01 00:00:00'],
            ['name' => 'Sally', 'deleted_at' => '2019-01-02 00:00:00'],
        ]
    */

<a name="method-wherestrict"></a>
#### `whereStrict()`

Этот метод имеет ту же сигнатуру, что и [`where`](#method-where) метод; однако все значения сравниваются с использованием «строгих» сравнений.

<a name="method-wherebetween"></a>
#### `whereBetween()`

Метод `whereBetween` фильтрует коллекцию, определяя, находится ли указанное значение элемента в заданном диапазоне:

    $collection = collect([
        ['product' => 'Desk', 'price' => 200],
        ['product' => 'Chair', 'price' => 80],
        ['product' => 'Bookcase', 'price' => 150],
        ['product' => 'Pencil', 'price' => 30],
        ['product' => 'Door', 'price' => 100],
    ]);

    $filtered = $collection->whereBetween('price', [100, 200]);

    $filtered->all();

    /*
        [
            ['product' => 'Desk', 'price' => 200],
            ['product' => 'Bookcase', 'price' => 150],
            ['product' => 'Door', 'price' => 100],
        ]
    */

<a name="method-wherein"></a>
#### `whereIn()`

Метод `whereIn` удаляет элементы из коллекции, у которых нет указанного значения элемента, содержащегося в данном массиве:

    $collection = collect([
        ['product' => 'Desk', 'price' => 200],
        ['product' => 'Chair', 'price' => 100],
        ['product' => 'Bookcase', 'price' => 150],
        ['product' => 'Door', 'price' => 100],
    ]);

    $filtered = $collection->whereIn('price', [150, 200]);

    $filtered->all();

    /*
        [
            ['product' => 'Desk', 'price' => 200],
            ['product' => 'Bookcase', 'price' => 150],
        ]
    */

Метод `whereIn` использует «свободные» сравнения при проверке значений элементов, то есть строка с целочисленным значением будет считаться равной целому числу с тем же значением. Используйте [`whereInStrict`](#method-whereinstrict) метод для фильтрации с использованием «строгих» сравнений.

<a name="method-whereinstrict"></a>
#### `whereInStrict()` 

Этот метод имеет ту же сигнатуру, что и [`whereIn`](#method-wherein) метод; однако все значения сравниваются с использованием «строгих» сравнений.

<a name="method-whereinstanceof"></a>
#### `whereInstanceOf()`

Метод `whereInstanceOf` фильтрует коллекцию по заданному типу класса:

    use App\Models\User;
    use App\Models\Post;

    $collection = collect([
        new User,
        new User,
        new Post,
    ]);

    $filtered = $collection->whereInstanceOf(User::class);

    $filtered->all();

    // [App\Models\User, App\Models\User]

<a name="method-wherenotbetween"></a>
#### `whereNotBetween()`

Метод `whereNotBetween` фильтрует коллекцию, определяя, находится ли указанное значение элемента за пределами заданного диапазона:

    $collection = collect([
        ['product' => 'Desk', 'price' => 200],
        ['product' => 'Chair', 'price' => 80],
        ['product' => 'Bookcase', 'price' => 150],
        ['product' => 'Pencil', 'price' => 30],
        ['product' => 'Door', 'price' => 100],
    ]);

    $filtered = $collection->whereNotBetween('price', [100, 200]);

    $filtered->all();

    /*
        [
            ['product' => 'Chair', 'price' => 80],
            ['product' => 'Pencil', 'price' => 30],
        ]
    */

<a name="method-wherenotin"></a>
#### `whereNotIn()`

Метод `whereNotIn` удаляет элементы из коллекции, которые имеют указанное значение элемента, содержащееся в данном массиве:

    $collection = collect([
        ['product' => 'Desk', 'price' => 200],
        ['product' => 'Chair', 'price' => 100],
        ['product' => 'Bookcase', 'price' => 150],
        ['product' => 'Door', 'price' => 100],
    ]);

    $filtered = $collection->whereNotIn('price', [150, 200]);

    $filtered->all();

    /*
        [
            ['product' => 'Chair', 'price' => 100],
            ['product' => 'Door', 'price' => 100],
        ]
    */

Метод `whereNotIn` использует «свободные» сравнения при проверке значений элементов, то есть строка с целочисленным значением будет считаться равной целому числу с тем же значением. Используйте [`whereNotInStrict`](#method-wherenotinstrict) метод для фильтрации с использованием «строгих» сравнений.

<a name="method-wherenotinstrict"></a>
#### `whereNotInStrict()`

Этот метод имеет ту же сигнатуру, что и [`whereNotIn`](#method-wherenotin) метод; однако все значения сравниваются с использованием «строгих» сравнений.

<a name="method-wherenotnull"></a>
#### `whereNotNull()`

Метод `whereNotNull` возвращает элементы из коллекции, где данный ключ не является `null`:

    $collection = collect([
        ['name' => 'Desk'],
        ['name' => null],
        ['name' => 'Bookcase'],
    ]);

    $filtered = $collection->whereNotNull('name');

    $filtered->all();

    /*
        [
            ['name' => 'Desk'],
            ['name' => 'Bookcase'],
        ]
    */

<a name="method-wherenull"></a>
#### `whereNull()`

Метод `whereNull` возвращает элементы из коллекции, где задан данный ключ `null`:

    $collection = collect([
        ['name' => 'Desk'],
        ['name' => null],
        ['name' => 'Bookcase'],
    ]);

    $filtered = $collection->whereNull('name');

    $filtered->all();

    /*
        [
            ['name' => null],
        ]
    */


<a name="method-wrap"></a>
#### `wrap()`

Статический `wrap` метод оборачивает заданное значение в коллекцию, когда это применимо:

    use Mnv\Core\Collections\Collection;

    $collection = Collection::wrap('John Doe');

    $collection->all();

    // ['John Doe']

    $collection = Collection::wrap(['John Doe']);

    $collection->all();

    // ['John Doe']

    $collection = Collection::wrap(collect('John Doe'));

    $collection->all();

    // ['John Doe']

<a name="method-zip"></a>

##### `zip()`

Метод `zip` объединяет значения данного массива со значениями исходной коллекции по их соответствующему индексу:

    $collection = collect(['Chair', 'Desk']);

    $zipped = $collection->zip([100, 200]);

    $zipped->all();

    // [['Chair', 100], ['Desk', 200]]

<a name="higher-order-messages"></a>
## Сообщения высшего порядка

Коллекции также обеспечивают поддержку «сообщений более высокого порядка», которые являются кратчайшим путем для выполнения общих действий с коллекциями. Методы сбора , которые предоставляют сообщения более высокого порядка:
[`average`](#method-average), 
[`avg`](#method-avg), 
[`contains`](#method-contains), 
[`each`](#method-each), 
[`every`](#method-every), 
[`filter`](#method-filter),
[`first`](#method-first),
[`flatMap`](#method-flatmap), 
[`groupBy`](#method-groupby), [`keyBy`](#method-keyby), [`map`](#method-map), [`max`](#method-max), [`min`](#method-min), [`partition`](#method-partition), [`reject`](#method-reject), [`skipUntil`](#method-skipuntil), [`skipWhile`](#method-skipwhile), [`some`](#method-some), [`sortBy`](#method-sortby), [`sortByDesc`](#method-sortbydesc), [`sum`](#method-sum), [`takeUntil`](#method-takeuntil), [`takeWhile`](#method-takewhile), and [`unique`](#method-unique).

К каждому сообщению более высокого порядка можно получить доступ как к динамическому свойству экземпляра коллекции. Например, давайте воспользуемся `each` сообщением более высокого порядка для вызова метода для каждого объекта в коллекции:

    use App\Models\User;

    $users = User::where('votes', '>', 500)->get();

    $users->each->markAsVip();

Точно так же мы можем использовать `sum` сообщение более высокого порядка для сбора общего количества «голосов» для набора пользователей:

    $users = User::where('group', 'Development')->get();

    return $users->sum->votes;

<a name="lazy-collections"></a>
## Ленивые коллекции

<a name="lazy-collection-introduction"></a>
### Введение

> **Предупреждение**  
> Прежде чем узнать больше о ленивых коллекциях, найдите время, чтобы ознакомиться с [PHP generators](https://www.php.net/manual/en/language.generators.overview.php).

Чтобы дополнить и без того мощный `Collection` класс, класс использует [generators](https://www.php.net/manual/en/language.generators.overview.php) LazyCollection PHP , чтобы вы могли работать с очень большими наборами данных, сохраняя низкое использование памяти.

Например, представьте, что вашему приложению необходимо обработать многогигабайтный файл журнала, используя методы сбора данных Laravel для анализа журналов. Вместо того, чтобы сразу считывать весь файл в память, ленивые коллекции могут использоваться для хранения в памяти только небольшой части файла в данный момент времени:

    use App\Models\LogEntry;
    use Mnv\Core\Collections\LazyCollection;

    LazyCollection::make(function () {
        $handle = fopen('log.txt', 'r');

        while (($line = fgets($handle)) !== false) {
            yield $line;
        }
    })->chunk(4)->map(function (array $lines) {
        return LogEntry::fromLines($lines);
    })->each(function (LogEntry $logEntry) {
        // Process the log entry...
    });

Или представьте, что вам нужно перебрать 10 000 моделей Eloquent. При использовании традиционных коллекций Laravel все 10 000 моделей Eloquent должны быть загружены в память одновременно:

    use App\Models\User;

    $users = User::all()->filter(function (User $user) {
        return $user->id > 500;
    });

Однако метод построителя запросов `cursor` возвращает `LazyCollection` экземпляр. Это позволяет вам по-прежнему выполнять только один запрос к базе данных, но при этом одновременно загружать в память только одну модель Eloquent. В этом примере `filter` обратный вызов не выполняется до тех пор, пока мы не обработаем каждого пользователя отдельно, что позволяет резко сократить использование памяти:

    use App\Models\User;

    $users = User::cursor()->filter(function (User $user) {
        return $user->id > 500;
    });

    foreach ($users as $user) {
        echo $user->id;
    }

<a name="creating-lazy-collections"></a>
### Создание ленивых коллекций

Чтобы создать ленивый экземпляр коллекции, вы должны передать функцию генератора PHP методу коллекции `make`:

    use Mnv\Core\Collections\LazyCollection;

    LazyCollection::make(function () {
        $handle = fopen('log.txt', 'r');

        while (($line = fgets($handle)) !== false) {
            yield $line;
        }
    });

<a name="the-enumerable-contract"></a>
### Перечислимый контракт

Почти все методы, доступные в `Collection` классе, также доступны в `LazyCollection` классе. Оба этих класса реализуют `Mnv\Core\CollectionsEnumerable` контракт, который определяет следующие методы:


[all](#method-all)
[average](#method-average)
[avg](#method-avg)
[chunk](#method-chunk)
[chunkWhile](#method-chunkwhile)
[collapse](#method-collapse)
[collect](#method-collect)
[combine](#method-combine)
[concat](#method-concat)
[contains](#method-contains)
[containsStrict](#method-containsstrict)
[count](#method-count)
[countBy](#method-countBy)
[crossJoin](#method-crossjoin)
[dd](#method-dd)
[diff](#method-diff)
[diffAssoc](#method-diffassoc)
[diffKeys](#method-diffkeys)
[dump](#method-dump)
[duplicates](#method-duplicates)
[duplicatesStrict](#method-duplicatesstrict)
[each](#method-each)
[eachSpread](#method-eachspread)
[every](#method-every)
[except](#method-except)
[filter](#method-filter)
[first](#method-first)
[firstOrFail](#method-first-or-fail)
[firstWhere](#method-first-where)
[flatMap](#method-flatmap)
[flatten](#method-flatten)
[flip](#method-flip)
[forPage](#method-forpage)
[get](#method-get)
[groupBy](#method-groupby)
[has](#method-has)
[implode](#method-implode)
[intersect](#method-intersect)
[intersectAssoc](#method-intersectAssoc)
[intersectByKeys](#method-intersectbykeys)
[isEmpty](#method-isempty)
[isNotEmpty](#method-isnotempty)
[join](#method-join)
[keyBy](#method-keyby)
[keys](#method-keys)
[last](#method-last)
[macro](#method-macro)
[make](#method-make)
[map](#method-map)
[mapInto](#method-mapinto)
[mapSpread](#method-mapspread)
[mapToGroups](#method-maptogroups)
[mapWithKeys](#method-mapwithkeys)
[max](#method-max)
[median](#method-median)
[merge](#method-merge)
[mergeRecursive](#method-mergerecursive)
[min](#method-min)
[mode](#method-mode)
[nth](#method-nth)
[only](#method-only)
[pad](#method-pad)
[partition](#method-partition)
[pipe](#method-pipe)
[pluck](#method-pluck)
[random](#method-random)
[reduce](#method-reduce)
[reject](#method-reject)
[replace](#method-replace)
[replaceRecursive](#method-replacerecursive)
[reverse](#method-reverse)
[search](#method-search)
[shuffle](#method-shuffle)
[skip](#method-skip)
[slice](#method-slice)
[sole](#method-sole)
[some](#method-some)
[sort](#method-sort)
[sortBy](#method-sortby)
[sortByDesc](#method-sortbydesc)
[sortKeys](#method-sortkeys)
[sortKeysDesc](#method-sortkeysdesc)
[split](#method-split)
[sum](#method-sum)
[take](#method-take)
[tap](#method-tap)
[times](#method-times)
[toArray](#method-toarray)
[toJson](#method-tojson)
[union](#method-union)
[unique](#method-unique)
[uniqueStrict](#method-uniquestrict)
[unless](#method-unless)
[unlessEmpty](#method-unlessempty)
[unlessNotEmpty](#method-unlessnotempty)
[unwrap](#method-unwrap)
[values](#method-values)
[when](#method-when)
[whenEmpty](#method-whenempty)
[whenNotEmpty](#method-whennotempty)
[where](#method-where)
[whereStrict](#method-wherestrict)
[whereBetween](#method-wherebetween)
[whereIn](#method-wherein)
[whereInStrict](#method-whereinstrict)
[whereInstanceOf](#method-whereinstanceof)
[whereNotBetween](#method-wherenotbetween)
[whereNotIn](#method-wherenotin)
[whereNotInStrict](#method-wherenotinstrict)
[wrap](#method-wrap)
[zip](#method-zip)


> **Предупреждение**  
> Методы, изменяющие коллекцию (например `shift`, `pop`, `prepend` и т. д.), недоступны в `LazyCollection` классе.

<a name="lazy-collection-methods"></a>
### Ленивые методы сбора

В дополнение к методам, определенным в `Enumerable` контракте, `LazyCollection` класс содержит следующие методы:

<a name="method-takeUntilTimeout"></a>
#### `takeUntilTimeout()`

Метод `takeUntilTimeout` возвращает новую ленивую коллекцию, которая будет перечислять значения до указанного времени. По истечении этого времени коллекция перестанет перечислять:

    $lazyCollection = LazyCollection::times(INF)
        ->takeUntilTimeout(now()->addMinute());

    $lazyCollection->each(function (int $number) {
        dump($number);

        sleep(1);
    });

    // 1
    // 2
    // ...
    // 58
    // 59

Чтобы проиллюстрировать использование этого метода, представьте себе приложение, которое отправляет счета из базы данных с помощью курсора. Вы можете определить [запланированную задачу](/docs/scheduling) , которая запускается каждые 15 минут и обрабатывает счета не более 14 минут:

    use App\Models\Invoice;
    use Mnv\Core\Collections\Carbon;

    Invoice::pending()->cursor()
        ->takeUntilTimeout(
            Carbon::createFromTimestamp(LARAVEL_START)->add(14, 'minutes')
        )
        ->each(fn (Invoice $invoice) => $invoice->submit());

<a name="method-tapEach"></a>
#### `tapEach()`

В то время как `each` метод сразу вызывает данный обратный вызов для каждого элемента в коллекции, `tapEach` метод вызывает данный обратный вызов только тогда, когда элементы извлекаются из списка один за другим:

    // Nothing has been dumped so far...
    $lazyCollection = LazyCollection::times(INF)->tapEach(function (int $value) {
        dump($value);
    });

    // Three items are dumped...
    $array = $lazyCollection->take(3)->all();

    // 1
    // 2
    // 3

<a name="method-remember"></a>
#### `remember()`

Метод `remember` возвращает новую ленивую коллекцию, которая запоминает все значения, которые уже были перечислены, и не будет извлекать их снова при последующих перечислениях коллекции:

    // No query has been executed yet...
    $users = User::cursor()->remember();

    // The query is executed...
    // The first 5 users are hydrated from the database...
    $users->take(5)->all();

    // First 5 users come from the collection's cache...
    // The rest are hydrated from the database...
    $users->take(20)->all();
